/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.tracer.exporter.config.impl;

import static org.mule.runtime.container.api.MuleFoldersUtil.getConfFolder;
import static org.mule.runtime.tracer.exporter.config.api.OpenTelemetrySpanExporterConfigurationProperties.MULE_OPEN_TELEMETRY_TRACING_CONFIGURATION_FILE_NAME;
import static org.mule.runtime.tracer.exporter.config.api.OpenTelemetrySpanExporterConfigurationProperties.MULE_OPEN_TELEMETRY_TRACING_CONFIGURATION_FILE_PATH;

import static java.lang.String.format;
import static java.lang.System.getProperty;
import static java.nio.file.Paths.get;
import static java.util.Optional.empty;

import org.mule.runtime.api.lifecycle.Disposable;
import org.mule.runtime.config.internal.model.dsl.config.DefaultConfigurationPropertiesResolver;
import org.mule.runtime.config.internal.model.dsl.config.SystemPropertiesConfigurationProvider;
import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.module.observability.AbstractFileObservabilitySignalConfiguration;
import org.mule.runtime.module.observability.configuration.ObservabilityConfigurationFileWatcher;
import org.mule.runtime.module.observability.configuration.ObservabilitySignalConfiguration;

import java.io.File;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * A {@link ObservabilitySignalConfiguration} based on a file in the conf folder.
 *
 * @since 4.5.0
 */
public class FileSpanExporterConfiguration extends AbstractFileObservabilitySignalConfiguration
    implements ObservabilitySignalConfiguration, Disposable {

  private static final Logger logger = LoggerFactory.getLogger(FileSpanExporterConfiguration.class);
  private final String configurationFileName =
      getProperty(MULE_OPEN_TELEMETRY_TRACING_CONFIGURATION_FILE_NAME, "tracer-exporter.conf");
  private final CompositeRunnable doOnConfigurationChanged = new CompositeRunnable();
  private ObservabilityConfigurationFileWatcher tracingConfigurationFileWatcher;
  private Path resolvedConfigurationFilePath;
  private boolean tracingConfigurationFileWatcherInitialised;
  private final String artifactId;


  public FileSpanExporterConfiguration(MuleContext muleContext) {
    super(path -> findArtifactConfigFile(muleContext.getExecutionClassLoader(), path),
          propertyReference -> new DefaultConfigurationPropertiesResolver(empty(),
                                                                          new SystemPropertiesConfigurationProvider())
                                                                              .apply(propertyReference));
    this.artifactId = muleContext.getId();
  }

  @Override
  protected Path getSignalConfigurationFileDirectoryPath() {
    if (resolvedConfigurationFilePath != null) {
      return resolvedConfigurationFilePath;
    }
    resolvedConfigurationFilePath = get(getProperty(MULE_OPEN_TELEMETRY_TRACING_CONFIGURATION_FILE_PATH,
                                                    getConfFolder().getAbsolutePath())).toAbsolutePath();
    return resolvedConfigurationFilePath;
  }

  protected String getSignalConfigurationFileName() {
    return configurationFileName;
  }

  @Override
  protected void initialise() {
    super.initialise();
    // TODO W-19296402: Generalize the configuration reloading logic.
    if (getConfigurationFile() != null && getConfigurationFile().exists() && !tracingConfigurationFileWatcherInitialised) {
      tracingConfigurationFileWatcher =
          new ObservabilityConfigurationFileWatcher(getConfigurationFile(), doOnConfigurationChanged);
      tracingConfigurationFileWatcher.start();
      tracingConfigurationFileWatcherInitialised = true;
    }
  }

  @Override
  protected void onConfigurationFileNotFound() {
    logger.atInfo().log("Tracing exporter configuration file named '{}' not found {}. Using default configuration.",
                        getSignalConfigurationFileName(), getConfigurationFileLocations());
  }

  private String getConfigurationFileLocations() {
    if (isApplicationLevelConfigurable()) {
      return format("at at both '%s' artifact resources and the '%s' configuration path",
                    artifactId, getSignalConfigurationFileDirectoryPath());
    } else {
      return format("at the '%s' configuration path", getSignalConfigurationFileDirectoryPath());
    }
  }

  @Override
  protected void onConfigurationFileLoadError(Exception error, File configurationFile) {
    logger.atWarn().log("Span exporter configuration file at '{}' had a parsing error: {}. Using default configuration.",
                        configurationFile.getAbsolutePath(), error.getMessage());
  }

  @Override
  public void doOnConfigurationChanged(Runnable doOnConfigurationChanged) {
    this.doOnConfigurationChanged.addRunnable(doOnConfigurationChanged);
  }

  @Override
  public void dispose() {
    if (tracingConfigurationFileWatcher != null) {
      tracingConfigurationFileWatcher.interrupt();
    }
  }

  private class CompositeRunnable implements Runnable {

    private final List<Runnable> runnables = new ArrayList<>();

    CompositeRunnable() {
      runnables.add(FileSpanExporterConfiguration.this::initialise);
    }

    @Override
    public void run() {
      runnables.forEach(Runnable::run);
    }

    public void addRunnable(Runnable runnable) {
      runnables.add(runnable);
    }

  }
}
