/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.oauth.api.state;

import org.mule.oauth.client.api.state.DancerState;
import org.mule.runtime.api.lock.LockFactory;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.locks.Lock;

import static org.mule.oauth.client.api.state.DancerState.HAS_TOKEN;
import static org.mule.oauth.client.api.state.DancerState.NO_TOKEN;
import static org.mule.oauth.client.api.state.DancerState.TOKEN_INVALIDATED;

/**
 * OAuth state for a particular resource owner which typically represents an user. Only used for compatibility purposes in case of
 * mule version upgrade (W-15595791)
 *
 * @deprecated since 1.5, use {@link org.mule.oauth.client.api.state.ResourceOwnerOAuthContextWithRefreshState} from
 *             {@code mule-oauth-client 2.x}.
 */
@Deprecated
public final class ResourceOwnerOAuthContextWithRefreshState implements ResourceOwnerOAuthContext, Serializable {

  private static final long serialVersionUID = 6118607567823801246L;

  private final String resourceOwnerId;
  private DancerState dancerState = NO_TOKEN;
  private String accessToken;
  private String refreshToken;
  private String state;
  private String expiresIn;
  private Map<String, Object> tokenResponseParameters = new HashMap<>();

  public ResourceOwnerOAuthContextWithRefreshState(final String resourceOwnerId) {
    this.resourceOwnerId = resourceOwnerId;
  }

  /**
   * Since the runtime can be upgraded when there are already contexts stored, previous versions must be migrated to the new one.
   * The newly added state cannot be added to the previous version because that would break the java serialization of those
   * objects.
   *
   * @param original the context to migrate
   */
  public ResourceOwnerOAuthContextWithRefreshState(final ResourceOwnerOAuthContext original) {
    this.resourceOwnerId = original.getResourceOwnerId();
    // By default, new instances are built with NO_TOKEN.
    // This initial value is for the case where the object was generated by a previous version where this attribute wasn't there.
    // In that case, the equivalent state for that is HAS_TOKEN.
    this.dancerState = original.getAccessToken() == null ? NO_TOKEN : HAS_TOKEN;
    this.accessToken = original.getAccessToken();
    this.refreshToken = original.getRefreshToken();
    this.state = original.getState();
    this.expiresIn = original.getExpiresIn();
    this.tokenResponseParameters.putAll(original.getTokenResponseParameters());
  }

  @Override
  public String getAccessToken() {
    return accessToken;
  }

  @Override
  public String getRefreshToken() {
    return refreshToken;
  }

  @Override
  public String getState() {
    return state;
  }

  public void setAccessToken(final String accessToken) {
    this.accessToken = accessToken;
  }

  public void setRefreshToken(final String refreshToken) {
    this.refreshToken = refreshToken;
  }

  public void setExpiresIn(final String expiresIn) {
    this.expiresIn = expiresIn;
  }

  @Override
  public String getExpiresIn() {
    return expiresIn;
  }

  public void setState(final String state) {
    this.state = state;
  }

  @Override
  public Map<String, Object> getTokenResponseParameters() {
    return tokenResponseParameters;
  }

  public void setTokenResponseParameters(final Map<String, Object> tokenResponseParameters) {
    this.tokenResponseParameters = tokenResponseParameters;
  }

  @Override
  public String getResourceOwnerId() {
    return resourceOwnerId == null ? DEFAULT_RESOURCE_OWNER_ID : resourceOwnerId;
  }

  @Override
  public DancerState getDancerState() {
    return dancerState;
  }

  @Override
  public void setDancerState(DancerState dancerState) {
    if (dancerState == NO_TOKEN) {
      this.accessToken = null;
    }
    this.dancerState = dancerState;
  }

  @Override
  public Lock getRefreshOAuthContextLock(String lockNamePrefix, LockFactory lockProvider) {
    return createRefreshOAuthContextLock(lockNamePrefix, lockProvider, resourceOwnerId);
  }

  @Override
  public boolean isTokenInvalid() {
    return getDancerState() == TOKEN_INVALIDATED;
  }

  @Override
  public void markTokenAsInvalid() {
    setDancerState(TOKEN_INVALIDATED);
  }

  /**
   * This utility method is needed for the cases where the context is being queried. There is still no context created, but the
   * same lock that this context would use is required for querying thread-safely.
   */
  public static Lock createRefreshOAuthContextLock(String lockNamePrefix, LockFactory lockProvider, String resourceOwnerId) {
    return lockProvider.createLock(lockNamePrefix + "_oauth:" + resourceOwnerId);
  }
}
