/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.troubleshooting.internal;

import static org.mule.runtime.api.component.ComponentIdentifier.buildFromStringRepresentation;
import static org.mule.runtime.api.component.TypedComponentIdentifier.ComponentType.OPERATION;
import static org.mule.runtime.core.api.config.FeatureFlaggingRegistry.getInstance;
import static org.mule.runtime.core.api.event.EventContextService.EventContextState.EXECUTING;
import static org.mule.runtime.module.troubleshooting.internal.TroubleshootingTestUtils.TestFeature.TEST_FEATURE_1;
import static org.mule.runtime.module.troubleshooting.internal.TroubleshootingTestUtils.TestFeature.TEST_FEATURE_2;

import static java.time.Clock.fixed;
import static java.time.Instant.now;
import static java.time.Instant.ofEpochMilli;
import static java.time.ZoneId.of;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.singletonList;
import static java.util.Collections.unmodifiableList;
import static java.util.Optional.of;

import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.mockStatic;
import static org.mockito.Mockito.when;
import static org.mockito.Mockito.withSettings;

import org.mule.runtime.api.alert.TimedDataAggregation;
import org.mule.runtime.api.artifact.Registry;
import org.mule.runtime.api.component.TypedComponentIdentifier;
import org.mule.runtime.api.component.location.ComponentLocation;
import org.mule.runtime.api.config.Feature;
import org.mule.runtime.api.scheduler.SchedulerService;
import org.mule.runtime.api.scheduler.SchedulerView;
import org.mule.runtime.api.service.ServiceRepository;
import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.core.api.config.FeatureFlaggingRegistry;
import org.mule.runtime.core.api.context.notification.FlowCallStack;
import org.mule.runtime.core.api.context.notification.FlowStackElement;
import org.mule.runtime.core.api.event.EventContextService;
import org.mule.runtime.core.api.event.EventContextService.FlowStackEntry;
import org.mule.runtime.core.api.management.stats.AllStatistics;
import org.mule.runtime.deployment.model.api.application.Application;
import org.mule.runtime.deployment.model.api.application.ApplicationStatus;
import org.mule.runtime.deployment.model.api.artifact.ArtifactContext;
import org.mule.runtime.module.deployment.api.DeploymentService;

import java.util.Optional;

import java.time.Instant;

import org.mockito.MockedStatic;

public final class TroubleshootingTestUtils {

  private TroubleshootingTestUtils() {}

  public static DeploymentService mockDeploymentService(Application... applications) {
    DeploymentService deploymentService = mock(DeploymentService.class);
    when(deploymentService.getApplications()).thenReturn(asList(applications));
    for (Application application : applications) {
      when(deploymentService.findApplication(eq(application.getArtifactName()))).thenReturn(application);
    }
    return deploymentService;
  }

  public static ServiceRepository mockServiceRepository() {
    final SchedulerView sch1 = mock(SchedulerView.class);
    when(sch1.getName()).thenReturn("Sch 1");
    when(sch1.getActiveTasks()).thenReturn(8);
    when(sch1.getThreadType()).thenReturn("CPU_LITE");
    when(sch1.getQueuedTasks()).thenReturn(35);
    when(sch1.getRejectionsOverTime()).thenReturn(new TimedDataAggregation<>(now(), 2, 6, 13, 24));
    when(sch1.getThrottlesOverTime()).thenReturn(new TimedDataAggregation<>(now(), 0, 0, 0, 0));
    final SchedulerView sch2 = mock(SchedulerView.class);
    when(sch2.getName()).thenReturn("Sch 2");
    when(sch2.getThreadType()).thenReturn("IO");
    when(sch2.getActiveTasks()).thenReturn(3);
    when(sch2.getQueuedTasks()).thenReturn(0);
    when(sch2.getRejectionsOverTime()).thenReturn(new TimedDataAggregation<>(now(), 0, 0, 0, 0));
    when(sch2.getThrottlesOverTime()).thenReturn(new TimedDataAggregation<>(now(), 2, 7, 14, 27));

    final SchedulerService schedulerService = mock(SchedulerService.class);
    when(schedulerService.getSchedulers()).thenReturn(unmodifiableList(asList(sch1, sch2)));

    final ServiceRepository serviceRepository = mock(ServiceRepository.class);
    when(serviceRepository.getServices()).thenReturn(singletonList(schedulerService));

    return serviceRepository;
  }

  public static Application mockApplication(String appName, FlowStackEntry... flowStackEntries) {
    Application mockApp = mock(Application.class);
    when(mockApp.getArtifactName()).thenReturn(appName);
    // Set a default status (CREATED is the initial state for applications)
    when(mockApp.getStatus()).thenReturn(ApplicationStatus.CREATED);

    EventContextService eventContextService = mock(EventContextService.class);
    when(eventContextService.getCurrentlyActiveFlowStacks()).thenReturn(asList(flowStackEntries));

    Registry registry = mock(Registry.class);
    when(registry.lookupByName(EventContextService.REGISTRY_KEY)).thenReturn(of(eventContextService));

    // Mock MuleContext with statistics
    MuleContext muleContext = mock(MuleContext.class);
    AllStatistics allStatistics = mock(AllStatistics.class);
    when(muleContext.getStatistics()).thenReturn(allStatistics);

    ArtifactContext artifactContext = mock(ArtifactContext.class);
    when(artifactContext.getRegistry()).thenReturn(registry);
    when(artifactContext.getMuleContext()).thenReturn(muleContext);
    when(mockApp.getArtifactContext()).thenReturn(artifactContext);
    return mockApp;
  }

  public static FlowStackEntry mockFlowStackEntry(String eventId) {
    FlowStackEntry mockEntry = mock(FlowStackEntry.class, withSettings().name("<FlowStackEntry>"));
    when(mockEntry.getEventId()).thenReturn(eventId);
    when(mockEntry.getServerId()).thenReturn("ServerId");
    when(mockEntry.getState()).thenReturn(EXECUTING);

    final TypedComponentIdentifier tci = TypedComponentIdentifier.builder()
        .identifier(buildFromStringRepresentation("ns:component"))
        .type(OPERATION)
        .build();
    final ComponentLocation location = mock(ComponentLocation.class);
    when(location.getComponentIdentifier()).thenReturn(tci);
    when(location.getLocation()).thenReturn("MockLocation");

    Instant instant = now(fixed(ofEpochMilli(0), of("UTC")));
    try (MockedStatic<Instant> mockedStatic = mockStatic(Instant.class)) {
      mockedStatic.when(Instant::now).thenReturn(instant);

      final FlowStackElement flowStackElement = new FlowStackElement("MockFlow", "MockLocation", location, emptyMap());

      FlowCallStack flowCallStack = mock(FlowCallStack.class, withSettings().name("<FlowCallStack>"));
      when(flowCallStack.getElements()).thenReturn(singletonList(flowStackElement));

      when(mockEntry.getFlowCallStack()).thenReturn(flowCallStack);
      return mockEntry;
    }
  }

  public static FlowStackEntry mockFlowStackEntry(String eventId, FlowStackEntry parent) {
    FlowStackEntry mockEntry = mockFlowStackEntry(eventId);
    final String parentEventId = parent.getEventId();
    when(mockEntry.getParentEventId()).thenReturn(parentEventId);
    return mockEntry;
  }

  /**
   * Test features for use in troubleshooting tests.
   */
  public enum TestFeature implements Feature {

    TEST_FEATURE_1("Test feature 1", "TEST-1", "4.11.0"), TEST_FEATURE_2("Test feature 2", "TEST-2", "4.11.0");

    private final String description;
    private final String issue;
    private final String since;

    TestFeature(String description, String issue, String since) {
      this.description = description;
      this.issue = issue;
      this.since = since;
    }

    @Override
    public String getDescription() {
      return description;
    }

    @Override
    public String getIssueId() {
      return issue;
    }

    @Override
    public String getSince() {
      return since;
    }

    @Override
    public String getEnabledByDefaultSince() {
      return since;
    }

    @Override
    public Optional<String> getOverridingSystemPropertyName() {
      return Optional.empty();
    }
  }

  /**
   * Registers test features in the FeatureFlaggingRegistry.
   */
  public static void registerTestFeatures() {
    FeatureFlaggingRegistry registry = getInstance();
    try {
      registry.registerFeatureFlag(TEST_FEATURE_1, featureContext -> true);
    } catch (IllegalArgumentException e) {
      // Already registered, ignore
    }
    try {
      registry.registerFeatureFlag(TEST_FEATURE_2, featureContext -> true);
    } catch (IllegalArgumentException e) {
      // Already registered, ignore
    }
  }
}
