/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.troubleshooting.internal.operations;

import static java.lang.String.format;
import static java.lang.System.lineSeparator;
import static java.time.Instant.ofEpochMilli;
import static java.time.LocalDateTime.ofInstant;
import static java.time.ZoneId.systemDefault;
import static java.time.format.DateTimeFormatter.ofPattern;
import static java.util.Collections.emptyList;
import static java.util.Collections.sort;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toList;

import static org.mule.runtime.core.api.config.FeatureFlaggingRegistry.getInstance;
import static org.mule.runtime.module.artifact.api.descriptor.DomainDescriptor.DEFAULT_DOMAIN_NAME;
import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.api.config.Feature;
import org.mule.runtime.api.config.FeatureFlaggingService;
import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.core.api.policy.PolicyProvider;
import org.mule.runtime.deployment.model.api.application.Application;
import org.mule.runtime.deployment.model.api.application.ApplicationStatus;
import org.mule.runtime.deployment.model.api.domain.DomainStatus;
import org.mule.runtime.module.deployment.impl.internal.application.MuleApplicationPolicyProvider;
import org.mule.runtime.deployment.model.api.artifact.ArtifactContext;
import org.mule.runtime.deployment.model.api.domain.Domain;
import org.mule.runtime.deployment.model.api.plugin.ArtifactPlugin;
import org.mule.runtime.module.artifact.api.descriptor.DeployableArtifactDescriptor;
import org.mule.runtime.deployment.model.api.policy.PolicyTemplate;
import org.mule.runtime.module.artifact.api.descriptor.ArtifactPluginDescriptor;
import org.mule.runtime.module.deployment.impl.internal.policy.ApplicationPolicyInstance;
import org.mule.runtime.module.artifact.api.descriptor.BundleDescriptor;
import org.mule.runtime.module.deployment.api.DeploymentService;
import org.mule.runtime.module.troubleshooting.api.TroubleshootingOperation;
import org.mule.runtime.module.troubleshooting.api.TroubleshootingOperationCallback;
import org.mule.runtime.module.troubleshooting.api.TroubleshootingOperationDefinition;
import org.mule.runtime.module.troubleshooting.internal.DefaultTroubleshootingOperationDefinition;
import org.slf4j.Logger;

import java.io.IOException;
import java.io.Writer;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Supplier;

/**
 * Operation used to collect information about all deployments: applications, domains and policies.
 * <p>
 * The name of the operation is "deployments".
 */
public class DeploymentsOperation implements TroubleshootingOperation {

  public static final String DEPLOYMENTS_OPERATION_NAME = "deployments";
  public static final String DEPLOYMENTS_OPERATION_DESCRIPTION =
      "Lists all current deployments: applications, domains and policies";

  private static final TroubleshootingOperationDefinition definition = createOperationDefinition();
  private static final Logger LOGGER = getLogger(DeploymentsOperation.class.getName());

  private final DeploymentService deploymentService;

  public DeploymentsOperation(DeploymentService deploymentService) {
    this.deploymentService = deploymentService;
  }

  @Override
  public TroubleshootingOperationDefinition getDefinition() {
    return definition;
  }

  @Override
  public TroubleshootingOperationCallback getCallback() {
    return (arguments, writer) -> {
      writeDeploymentsInfo(writer);
    };
  }

  private void writeDeploymentsInfo(Writer writer) throws IOException {
    Map<String, PolicyTemplateInfo> policyTemplates = new LinkedHashMap<>();
    List<DeploymentInfo> deployments = collectDeployments(policyTemplates);

    if (deployments.isEmpty() && policyTemplates.isEmpty()) {
      writer.write("No deployments found." + lineSeparator());
      return;
    }

    // Write applications and domains
    for (DeploymentInfo deployment : deployments) {
      writeDeploymentInfo(deployment, writer);
      writer.write(lineSeparator());
    }

    // Write policy templates
    if (!policyTemplates.isEmpty()) {
      for (PolicyTemplateInfo template : policyTemplates.values()) {
        writePolicyTemplateInfo(template, writer);
        writer.write(lineSeparator());
      }
    }
  }

  private List<DeploymentInfo> collectDeployments(Map<String, PolicyTemplateInfo> policyTemplates) {
    List<DeploymentInfo> deployments = new ArrayList<>();

    // Collect applications with their policies and templates
    for (Application app : deploymentService.getApplications()) {
      List<PolicyInfo> policies = collectPoliciesFromApplication(app, policyTemplates);
      deployments.add(createApplicationDeploymentInfo(app, policies));
    }

    // Collect domains
    for (Domain domain : deploymentService.getDomains()) {
      deployments.add(createDomainDeploymentInfo(domain));
    }

    return deployments;
  }

  private List<PolicyInfo> collectPoliciesFromApplication(Application app,
                                                          Map<String, PolicyTemplateInfo> policyTemplates) {
    List<PolicyInfo> policies = new ArrayList<>();

    getMuleApplicationPolicyProvider(app)
        .ifPresent(policyProvider -> policies.addAll(extractPoliciesFromProvider(policyProvider, policyTemplates)));

    return policies;
  }

  private Optional<MuleApplicationPolicyProvider> getMuleApplicationPolicyProvider(Application app) {
    try {
      if (app.getArtifactContext() != null && app.getArtifactContext().getRegistry() != null) {
        Optional<PolicyProvider> policyProviderOpt =
            app.getArtifactContext().getRegistry().lookupByType(PolicyProvider.class);

        if (policyProviderOpt.isPresent() && policyProviderOpt.get() instanceof MuleApplicationPolicyProvider muleProvider) {
          return of(muleProvider);
        }
      }
    } catch (Exception e) {
      LOGGER.debug("Unable to access policy provider for application '{}'", app.getArtifactName(), e);
    }
    return empty();
  }

  private List<PolicyInfo> extractPoliciesFromProvider(MuleApplicationPolicyProvider provider,
                                                       Map<String, PolicyTemplateInfo> policyTemplates) {
    try {
      return provider.getRegisteredPolicyInstanceProviders().stream()
          .filter(providerObj -> providerObj.getPolicyId() != null)
          .map(providerObj -> {
            String policyId = providerObj.getPolicyId();

            // Add template to the map if not already present
            // Extract the unique template identifier (groupId:artifactId:version) from the full artifactId
            // The full artifactId includes the application path, so we need to extract just the template part
            ofNullable(providerObj.getApplicationPolicyInstance())
                .map(ApplicationPolicyInstance::getPolicyTemplate)
                .ifPresent(policyTemplate -> {
                  String uniqueKey = extractTemplateUniqueKey(policyTemplate);
                  if (uniqueKey != null && !policyTemplates.containsKey(uniqueKey)) {
                    String templateName = policyTemplate.getArtifactName();
                    String templateVersion = extractTemplateVersion(policyTemplate);
                    List<PluginInfo> plugins = extractPluginsFromPolicyTemplate(policyTemplate);
                    policyTemplates.put(uniqueKey,
                                        new PolicyTemplateInfo(uniqueKey, templateName, templateVersion, plugins));
                  }
                });

            return new PolicyInfo(policyId);
          })
          .collect(toList());
    } catch (Exception e) {
      LOGGER.debug("Unable to extract policies from provider", e);
      return emptyList();
    }
  }

  private String extractTemplateUniqueKey(PolicyTemplate template) {
    if (template.getDescriptor() != null && template.getDescriptor().getBundleDescriptor() != null) {
      BundleDescriptor bundleDesc = template.getDescriptor().getBundleDescriptor();
      // Build unique key as groupId:artifactId:version (same format as used in PolicyTemplateClassLoaderBuilder)
      return bundleDesc.getGroupId() + ":" + bundleDesc.getArtifactId() + ":" + bundleDesc.getVersion();
    }
    return null;
  }

  private String extractTemplateVersion(PolicyTemplate template) {
    if (template.getDescriptor() != null && template.getDescriptor().getBundleDescriptor() != null) {
      return template.getDescriptor().getBundleDescriptor().getVersion();
    }
    return "unknown";
  }

  private List<PluginInfo> extractPluginsFromPolicyTemplate(PolicyTemplate policyTemplate) {
    List<PluginInfo> plugins = new ArrayList<>();

    try {
      List<ArtifactPlugin> artifactPlugins = policyTemplate.getArtifactPlugins();
      if (artifactPlugins != null) {
        for (ArtifactPlugin plugin : artifactPlugins) {
          String pluginName = plugin.getDescriptor().getName();
          String version = extractPluginVersion(plugin.getDescriptor());
          plugins.add(new PluginInfo(pluginName, version));
        }
      }
    } catch (Exception e) {
      LOGGER.debug("Unable to extract plugins from policy template", e);
    }

    plugins.sort((p1, p2) -> p1.name.compareTo(p2.name));

    return plugins;
  }

  private DeploymentInfo createApplicationDeploymentInfo(Application app, List<PolicyInfo> policies) {
    String artifactName = app.getArtifactName();
    String artifactType = "APPLICATION";
    String state = mapApplicationStatus(app.getStatus());

    // Extract domain name (only if not default)
    String domainName = null;
    Domain domain = app.getDomain();
    if (domain != null) {
      String domainArtifactName = domain.getArtifactName();
      if (domainArtifactName != null && !DEFAULT_DOMAIN_NAME.equals(domainArtifactName)) {
        domainName = domainArtifactName;
      }
    }

    List<PluginInfo> plugins = extractPlugins(app);
    List<String> featureFlags = extractFeatureFlags(app);
    String lastStartTime = calculateLastStartTime(app.getArtifactContext(), artifactName);

    return new DeploymentInfo(artifactName, artifactType, state, plugins, featureFlags, lastStartTime, policies, domainName);
  }

  private DeploymentInfo createDomainDeploymentInfo(Domain domain) {
    String artifactName = domain.getArtifactName();
    String artifactType = "DOMAIN";
    String state = mapDomainStatus(domain.getStatus());

    List<PluginInfo> plugins = extractPlugins(domain);
    List<String> featureFlags = emptyList(); // Domains don't have feature flags in the same way
    String lastStartTime = calculateLastStartTime(domain.getArtifactContext(), artifactName);

    return new DeploymentInfo(artifactName, artifactType, state, plugins, featureFlags, lastStartTime, emptyList(), null);
  }

  private String mapApplicationStatus(ApplicationStatus status) {
    return switch (status) {
      case CREATED -> "INITIAL";
      case INITIALISED -> "DEPLOYED";
      case STARTED -> "STARTED";
      case STOPPED -> "STOPPED";
      case DEPLOYMENT_FAILED -> "FAILED";
      case DESTROYED -> "UNDEPLOYED";
    };
  }

  private String mapDomainStatus(DomainStatus status) {
    return switch (status) {
      case CREATED -> "INITIAL";
      case INITIALISED -> "DEPLOYED";
      case STARTED -> "STARTED";
      case STOPPED -> "STOPPED";
      case DEPLOYMENT_FAILED -> "FAILED";
      case DESTROYED -> "UNDEPLOYED";
    };
  }

  private List<PluginInfo> extractPlugins(Application app) {
    return extractPlugins(app::getArtifactPlugins, app::getDescriptor);
  }

  private List<PluginInfo> extractPlugins(Domain domain) {
    return extractPlugins(domain::getArtifactPlugins, domain::getDescriptor);
  }

  private List<PluginInfo> extractPlugins(Supplier<List<ArtifactPlugin>> artifactPluginsSupplier,
                                          Supplier<DeployableArtifactDescriptor> descriptorSupplier) {
    List<PluginInfo> plugins = new ArrayList<>();

    try {
      // Try to get plugins from artifact plugins first
      List<ArtifactPlugin> artifactPlugins = artifactPluginsSupplier.get();
      if (artifactPlugins != null) {
        for (ArtifactPlugin plugin : artifactPlugins) {
          String pluginName = plugin.getDescriptor().getName();
          String version = extractPluginVersion(plugin.getDescriptor());
          plugins.add(new PluginInfo(pluginName, version));
        }
      }

      // Fallback: try to get plugins from descriptor
      if (plugins.isEmpty()) {
        addPluginsFromDescriptor(plugins, descriptorSupplier.get());
      }
    } catch (Exception e) {
      // If artifact is undeployed, we might not be able to access plugins
      // In that case, try to get from descriptor if available
      LOGGER.debug("Unable to extract plugins from artifact, trying descriptor", e);
      addPluginsFromDescriptor(plugins, descriptorSupplier.get());
    }

    // Sort plugins by name
    sort(plugins, (p1, p2) -> p1.name.compareTo(p2.name));

    return plugins;
  }

  private void addPluginsFromDescriptor(List<PluginInfo> plugins, DeployableArtifactDescriptor descriptor) {
    if (descriptor != null) {
      try {
        Set<ArtifactPluginDescriptor> descriptorPlugins = descriptor.getPlugins();
        if (descriptorPlugins != null) {
          for (ArtifactPluginDescriptor pluginDesc : descriptorPlugins) {
            String pluginName = pluginDesc.getName();
            String version = extractPluginVersion(pluginDesc);
            plugins.add(new PluginInfo(pluginName, version));
          }
        }
      } catch (Exception e) {
        LOGGER.debug("Unable to extract plugin information from descriptor '{}'", descriptor.getName(), e);
      }
    }
  }

  private String extractPluginVersion(ArtifactPluginDescriptor pluginDesc) {
    BundleDescriptor bundleDesc = pluginDesc.getBundleDescriptor();
    if (bundleDesc != null) {
      return bundleDesc.getVersion();
    }
    return "unknown";
  }

  private List<String> extractFeatureFlags(Application app) {
    List<String> enabledFeatures = new ArrayList<>();

    try {
      // Only extract feature flags for DEPLOYED/STARTED/STOPPED artifacts
      ApplicationStatus status = app.getStatus();
      if (status != ApplicationStatus.DEPLOYMENT_FAILED && status != ApplicationStatus.DESTROYED) {
        FeatureFlaggingService featureFlaggingService = app.getArtifactContext()
            .getRegistry()
            .lookupByType(FeatureFlaggingService.class)
            .orElse(null);

        if (featureFlaggingService != null) {
          // Get all registered features from the registry
          Map<Feature, ?> registeredFeatures = getInstance().getFeatureFlagConfigurations();

          for (Feature feature : registeredFeatures.keySet()) {
            try {
              if (featureFlaggingService.isEnabled(feature)) {
                enabledFeatures.add(feature.name());
              }
            } catch (Exception e) {
              LOGGER.debug("Unable to check if feature '{}' is enabled", feature.name(), e);
            }
          }
        }
      }
    } catch (Exception e) {
      LOGGER.debug("Unable to extract feature flags from application '{}'", app.getArtifactName(), e);
    }

    // Sort alphabetically
    sort(enabledFeatures);

    return enabledFeatures;
  }

  private String calculateLastStartTime(ArtifactContext artifactContext, String deploymentName) {
    try {
      // Use MuleContext startDate if available
      if (artifactContext != null) {
        MuleContext muleContext = artifactContext.getMuleContext();
        if (muleContext != null) {
          long startDate = muleContext.getStartDate();
          // getStartDate() returns 0 if the context hasn't been started yet
          if (startDate > 0) {
            return formatTimestamp(startDate);
          }
        }
      }
    } catch (Exception e) {
      LOGGER.debug("Unable to calculate last start time for deployment '{}'", deploymentName, e);
    }

    return "unknown";
  }

  private String formatTimestamp(long timestampMillis) {
    LocalDateTime dateTime = ofInstant(ofEpochMilli(timestampMillis), systemDefault());
    return dateTime.format(ofPattern("yyyy-MM-dd HH:mm:ss"));
  }

  private void writeDeploymentInfo(DeploymentInfo deployment, Writer writer) throws IOException {
    writer.write(format("\"%s\" - type: %s, state: %s",
                        deployment.artifactName,
                        deployment.artifactType,
                        deployment.state));
    writer.write(lineSeparator());

    // Write domain (only for applications, only if not default)
    if (deployment.domainName != null && "APPLICATION".equals(deployment.artifactType)) {
      writer.write("  Domain: " + deployment.domainName);
      writer.write(lineSeparator());
    }

    writer.write("  Last started: " + deployment.timeSinceLastUpdate);
    writer.write(lineSeparator());

    // Write policies (only for applications) - only show policy ID
    if (!deployment.policies.isEmpty() && "APPLICATION".equals(deployment.artifactType)) {
      writer.write("  Policies:");
      writer.write(lineSeparator());
      for (PolicyInfo policy : deployment.policies) {
        writer.write("    - " + policy.id);
        writer.write(lineSeparator());
      }
    }

    // Write plugins
    if (!deployment.plugins.isEmpty()) {
      writer.write("  Plugins:");
      writer.write(lineSeparator());
      for (PluginInfo plugin : deployment.plugins) {
        writer.write(format("    - %s : %s", plugin.name, plugin.version));
        writer.write(lineSeparator());
      }
    }

    // Write feature flags (only for DEPLOYED/STARTED/STOPPED artifacts)
    if (!deployment.featureFlags.isEmpty() &&
        (deployment.state.equals("DEPLOYED") ||
            deployment.state.equals("STARTED") ||
            deployment.state.equals("STOPPED"))) {
      writer.write("  Enabled Feature Flags:");
      writer.write(lineSeparator());
      for (String featureFlag : deployment.featureFlags) {
        writer.write("    - " + featureFlag);
        writer.write(lineSeparator());
      }
    }
  }

  private void writePolicyTemplateInfo(PolicyTemplateInfo template, Writer writer) throws IOException {
    writer.write(format("\"%s\" - type: POLICY_TEMPLATE, version: %s",
                        template.name != null ? template.name : template.id,
                        template.version));
    writer.write(lineSeparator());

    // Write plugins
    if (!template.plugins.isEmpty()) {
      writer.write("  Plugins:");
      writer.write(lineSeparator());
      for (PluginInfo plugin : template.plugins) {
        writer.write(format("    - %s : %s", plugin.name, plugin.version));
        writer.write(lineSeparator());
      }
    }
  }

  private static TroubleshootingOperationDefinition createOperationDefinition() {
    return new DefaultTroubleshootingOperationDefinition(DEPLOYMENTS_OPERATION_NAME,
                                                         DEPLOYMENTS_OPERATION_DESCRIPTION);
  }

  private record DeploymentInfo(String artifactName, String artifactType, String state, List<PluginInfo> plugins,
                                List<String> featureFlags, String timeSinceLastUpdate, List<PolicyInfo> policies,
                                String domainName) {

  }


  private record PolicyInfo(String id) {

  }


  private record PolicyTemplateInfo(String id, String name, String version, List<PluginInfo> plugins) {

  }


  private record PluginInfo(String name, String version) {

  }
}

