/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.troubleshooting.internal.operations;

import static org.mule.runtime.core.api.config.FeatureFlaggingRegistry.getInstance;
import static org.mule.runtime.deployment.model.api.application.ApplicationStatus.CREATED;
import static org.mule.runtime.deployment.model.api.application.ApplicationStatus.DEPLOYMENT_FAILED;
import static org.mule.runtime.deployment.model.api.application.ApplicationStatus.DESTROYED;
import static org.mule.runtime.deployment.model.api.application.ApplicationStatus.INITIALISED;
import static org.mule.runtime.deployment.model.api.application.ApplicationStatus.STARTED;
import static org.mule.runtime.deployment.model.api.application.ApplicationStatus.STOPPED;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptyMap;
import static java.util.Collections.emptySet;
import static java.util.Set.of;

import static org.hamcrest.CoreMatchers.equalTo;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.not;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsString;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import java.io.File;
import java.io.IOException;
import java.io.StringWriter;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.mule.runtime.api.config.Feature;
import org.mule.runtime.api.config.FeatureFlaggingService;
import org.mule.runtime.api.artifact.Registry;
import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.core.api.policy.PolicyProvider;
import org.mule.runtime.core.api.policy.PolicyParametrization;
import org.mule.runtime.deployment.model.api.application.Application;
import org.mule.runtime.deployment.model.api.policy.PolicyTemplateDescriptor;
import org.mule.runtime.module.artifact.api.classloader.ArtifactClassLoader;
import org.mule.runtime.module.artifact.api.descriptor.BundleDescriptor.Builder;
import org.mule.runtime.module.deployment.impl.internal.application.MuleApplicationPolicyProvider;
import org.mule.runtime.module.deployment.impl.internal.policy.ApplicationPolicyInstance;
import org.mule.runtime.module.deployment.impl.internal.policy.PolicyInstanceProviderFactory;
import org.mule.runtime.module.deployment.impl.internal.policy.PolicyTemplateFactory;
import org.mule.runtime.policy.api.PolicyPointcut;

import static org.mule.runtime.module.troubleshooting.internal.TroubleshootingTestUtils.TestFeature.TEST_FEATURE_1;
import static org.mule.runtime.module.troubleshooting.internal.TroubleshootingTestUtils.TestFeature.TEST_FEATURE_2;
import static org.mule.runtime.module.troubleshooting.internal.TroubleshootingTestUtils.registerTestFeatures;

import org.mule.runtime.deployment.model.api.DeployableArtifact;
import org.mule.runtime.deployment.model.api.application.ApplicationStatus;
import org.mule.runtime.deployment.model.api.artifact.ArtifactContext;
import org.mule.runtime.deployment.model.api.domain.Domain;
import org.mule.runtime.deployment.model.api.domain.DomainStatus;
import org.mule.runtime.deployment.model.api.policy.PolicyTemplate;
import org.mule.runtime.deployment.model.api.plugin.ArtifactPlugin;
import org.mule.runtime.module.artifact.api.descriptor.ApplicationDescriptor;
import org.mule.runtime.module.artifact.api.descriptor.ArtifactPluginDescriptor;
import org.mule.runtime.module.artifact.api.descriptor.BundleDescriptor;
import org.mule.runtime.module.artifact.api.descriptor.DomainDescriptor;
import org.mule.runtime.module.deployment.api.DeploymentService;

class DeploymentsOperationTestCase {

  private DeploymentService deploymentService;
  private DeploymentsOperation deploymentsOperation;

  @BeforeEach
  public void setup() {
    deploymentService = mock(DeploymentService.class);
    deploymentsOperation = new DeploymentsOperation(deploymentService);

    // Register test features in the registry
    registerTestFeatures();
  }

  @Test
  void testDeploymentsOperationDefinition() {
    assertThat(deploymentsOperation.getDefinition().getName(), is("deployments"));
    assertThat(deploymentsOperation.getDefinition().getDescription(),
               is("Lists all current deployments: applications, domains and policies"));
    assertThat(deploymentsOperation.getDefinition().getArgumentDefinitions(), is(emptyList()));
  }

  @Test
  void testDeploymentsWithNoDeployments() throws IOException {
    when(deploymentService.getApplications()).thenReturn(emptyList());
    when(deploymentService.getDomains()).thenReturn(emptyList());

    StringWriter writer = new StringWriter();
    deploymentsOperation.getCallback().execute(emptyMap(), writer);
    String result = writer.toString();

    assertThat(result.replaceAll("\\r\\n", "\n"), is(equalTo("No deployments found.\n")));
  }

  @Test
  void testDeploymentsWithApplications() throws IOException {
    Application app1 = createMockApplication("my-app", STARTED);
    Application app2 = createMockApplication("test-app", STOPPED);

    when(deploymentService.getApplications()).thenReturn(asList(app1, app2));
    when(deploymentService.getDomains()).thenReturn(emptyList());

    StringWriter writer = new StringWriter();
    deploymentsOperation.getCallback().execute(emptyMap(), writer);
    String result = writer.toString();

    assertThat(result, containsString("\"my-app\" - type: APPLICATION, state: STARTED"));
    assertThat(result, containsString("\"test-app\" - type: APPLICATION, state: STOPPED"));
    assertThat(result, containsString("Last started:"));
  }

  @Test
  void testDeploymentsWithDifferentStates() throws IOException {
    Application appCreated = createMockApplication("app-created", CREATED);
    Application appInitialised = createMockApplication("app-initialised", INITIALISED);
    Application appStarted = createMockApplication("app-started", STARTED);
    Application appStopped = createMockApplication("app-stopped", STOPPED);
    Application appFailed = createMockApplication("app-failed", DEPLOYMENT_FAILED);
    Application appDestroyed = createMockApplication("app-destroyed", DESTROYED);

    Domain domainInitial = createMockDomain("domain-initial", DomainStatus.CREATED);
    Domain domainDeployed = createMockDomain("domain-deployed", DomainStatus.INITIALISED);
    Domain domainStarted = createMockDomain("domain-started", DomainStatus.STARTED);
    Domain domainStopped = createMockDomain("domain-stopped", DomainStatus.STOPPED);
    Domain domainFailed = createMockDomain("domain-failed", DomainStatus.DEPLOYMENT_FAILED);
    Domain domainUndeployed = createMockDomain("domain-undeployed", DomainStatus.DESTROYED);

    when(deploymentService.getApplications())
        .thenReturn(asList(appCreated, appInitialised, appStarted, appStopped, appFailed, appDestroyed));
    when(deploymentService.getDomains())
        .thenReturn(asList(domainInitial, domainDeployed, domainStarted, domainStopped, domainFailed, domainUndeployed));

    StringWriter writer = new StringWriter();
    deploymentsOperation.getCallback().execute(emptyMap(), writer);
    String result = writer.toString();

    // Verify application states
    assertThat(result, containsString("\"app-created\" - type: APPLICATION, state: INITIAL"));
    assertThat(result, containsString("\"app-initialised\" - type: APPLICATION, state: DEPLOYED"));
    assertThat(result, containsString("\"app-started\" - type: APPLICATION, state: STARTED"));
    assertThat(result, containsString("\"app-stopped\" - type: APPLICATION, state: STOPPED"));
    assertThat(result, containsString("\"app-failed\" - type: APPLICATION, state: FAILED"));
    assertThat(result, containsString("\"app-destroyed\" - type: APPLICATION, state: UNDEPLOYED"));

    // Verify domain states
    assertThat(result, containsString("\"domain-initial\" - type: DOMAIN, state: INITIAL"));
    assertThat(result, containsString("\"domain-deployed\" - type: DOMAIN, state: DEPLOYED"));
    assertThat(result, containsString("\"domain-started\" - type: DOMAIN, state: STARTED"));
    assertThat(result, containsString("\"domain-stopped\" - type: DOMAIN, state: STOPPED"));
    assertThat(result, containsString("\"domain-failed\" - type: DOMAIN, state: FAILED"));
    assertThat(result, containsString("\"domain-undeployed\" - type: DOMAIN, state: UNDEPLOYED"));
  }

  @Test
  void testDeploymentsWithPlugins() throws IOException {
    Application app = createMockApplicationWithPlugins();

    when(deploymentService.getApplications()).thenReturn(asList(app));
    when(deploymentService.getDomains()).thenReturn(emptyList());

    StringWriter writer = new StringWriter();
    deploymentsOperation.getCallback().execute(emptyMap(), writer);
    String result = writer.toString();

    assertThat(result, containsString("Plugins:"));
    assertThat(result, containsString("- HTTP : 1.10.3"));
    assertThat(result, containsString("- Sockets : 1.2.5"));
    assertThat(result, containsString("- APIKit : 1.11.3"));
  }

  @Test
  void testDeploymentsWithFeatureFlags() throws IOException {
    Application app = createMockApplicationWithSpecificFeatureFlags(
                                                                    TEST_FEATURE_1.name(),
                                                                    TEST_FEATURE_2.name());

    when(deploymentService.getApplications()).thenReturn(asList(app));
    when(deploymentService.getDomains()).thenReturn(emptyList());

    StringWriter writer = new StringWriter();
    deploymentsOperation.getCallback().execute(emptyMap(), writer);
    String result = writer.toString();

    // Verify the application appears with STARTED state
    assertThat(result, containsString("\"app-with-features\" - type: APPLICATION, state: STARTED"));

    // Verify that feature flags section appears with the expected flags
    assertThat(result, containsString("Enabled Feature Flags:"));
    assertThat(result, containsString("    - TEST_FEATURE_1"));
    assertThat(result, containsString("    - TEST_FEATURE_2"));
  }

  @Test
  void testDeploymentsWithApplicationAndNonDefaultDomain() throws IOException {
    Application app = createMockApplication("payment-service", STARTED);
    Domain sharedDomain = createMockDomain("shared-domain", DomainStatus.INITIALISED);

    // Mock app to return the shared domain
    when(app.getDomain()).thenReturn(sharedDomain);

    when(deploymentService.getApplications()).thenReturn(asList(app));
    when(deploymentService.getDomains()).thenReturn(asList(sharedDomain));

    StringWriter writer = new StringWriter();
    deploymentsOperation.getCallback().execute(emptyMap(), writer);
    String result = writer.toString();

    // Verify application shows domain
    assertThat(result, containsString("\"payment-service\" - type: APPLICATION, state: STARTED"));
    assertThat(result, containsString("  Domain: shared-domain"));
    assertThat(result, containsString("\"shared-domain\" - type: DOMAIN, state: DEPLOYED"));
  }

  @Test
  void testDeploymentsWithApplicationAndDefaultDomain() throws IOException {
    Application app = createMockApplication("inventory-service", STOPPED);
    Domain defaultDomain = createMockDomain("default", DomainStatus.INITIALISED);

    // Mock app to return the default domain
    when(app.getDomain()).thenReturn(defaultDomain);

    when(deploymentService.getApplications()).thenReturn(asList(app));
    when(deploymentService.getDomains()).thenReturn(emptyList());

    StringWriter writer = new StringWriter();
    deploymentsOperation.getCallback().execute(emptyMap(), writer);
    String result = writer.toString();

    // Verify application does NOT show domain (because it's default)
    assertThat(result, containsString("\"inventory-service\" - type: APPLICATION, state: STOPPED"));
    assertThat(result, not(containsString("  Domain: default")));
    assertThat(result, not(containsString("\"default\" - type: DOMAIN")));
  }

  @Test
  void testDeploymentsWithUndeployedApplication() throws IOException {
    // Application that is destroyed but still in memory (memory leak scenario)
    Application appDestroyed = createMockApplication("app-undeployed", DESTROYED);
    // Make getArtifactContext return null to simulate undeployed state
    when(appDestroyed.getArtifactContext()).thenReturn(null);

    when(deploymentService.getApplications()).thenReturn(asList(appDestroyed));
    when(deploymentService.getDomains()).thenReturn(emptyList());

    StringWriter writer = new StringWriter();
    deploymentsOperation.getCallback().execute(emptyMap(), writer);
    String result = writer.toString();

    // Should still list the undeployed application
    assertThat(result, containsString("\"app-undeployed\" - type: APPLICATION, state: UNDEPLOYED"));
  }

  @Test
  void testDeploymentsWithPolicies() throws Exception {
    Application app = createMockApplicationWithPolicies();

    when(deploymentService.getApplications()).thenReturn(asList(app));
    when(deploymentService.getDomains()).thenReturn(emptyList());

    StringWriter writer = new StringWriter();
    deploymentsOperation.getCallback().execute(emptyMap(), writer);
    String result = writer.toString();

    // Verify application appears
    assertThat(result, containsString("\"app-with-policies\" - type: APPLICATION, state: STARTED"));
    // Verify policies appear within the application (only IDs)
    assertThat(result, containsString("Policies:"));
    assertThat(result, containsString("    - policy-1"));
    assertThat(result, containsString("    - policy-2"));
    // Verify policy templates appear separately with id, version and plugins
    assertThat(result, containsString("type: POLICY_TEMPLATE, version:"));
    assertThat(result, containsString("  Plugins:"));
    assertThat(result, containsString("    - PolicyPlugin : 1.0.0"));
    assertThat(result, containsString("    - PolicyPlugin : 2.0.0"));
    // Verify policies appear after Last started
    int lastStartedIndex = result.indexOf("Last started:");
    int policiesIndex = result.indexOf("Policies:");
    assertThat("Policies should appear after Last started", lastStartedIndex < policiesIndex);
  }

  private Application createMockApplication(String name, ApplicationStatus status) {
    Application app = createMockDeployableArtifact(Application.class, name);
    when(app.getStatus()).thenReturn(status);

    // Mock descriptor
    ApplicationDescriptor descriptor = mock(ApplicationDescriptor.class);
    when(app.getDescriptor()).thenReturn(descriptor);
    when(descriptor.getPlugins()).thenReturn(emptySet());

    ArtifactContext artifactContext = app.getArtifactContext();

    // Mock MuleContext with startDate for started/stopped applications
    if (status == STARTED || status == STOPPED) {
      MuleContext muleContext = mock(MuleContext.class);
      when(artifactContext.getMuleContext()).thenReturn(muleContext);
    }

    return app;
  }

  private Application createMockApplicationWithPlugins() {
    Application app = createMockApplication("app-with-plugins", STARTED);

    // Create mock plugins
    ArtifactPlugin plugin1 = createMockPlugin("HTTP", "1.10.3");
    ArtifactPlugin plugin2 = createMockPlugin("Sockets", "1.2.5");
    ArtifactPlugin plugin3 = createMockPlugin("APIKit", "1.11.3");

    when(app.getArtifactPlugins()).thenReturn(asList(plugin1, plugin2, plugin3));

    return app;
  }

  private ArtifactPlugin createMockPlugin(String pluginName, String version) {
    ArtifactPlugin plugin = mock(ArtifactPlugin.class);
    ArtifactPluginDescriptor descriptor = mock(ArtifactPluginDescriptor.class);
    BundleDescriptor bundleDescriptor = mock(BundleDescriptor.class);

    when(plugin.getDescriptor()).thenReturn(descriptor);
    when(descriptor.getName()).thenReturn(pluginName);
    when(descriptor.getBundleDescriptor()).thenReturn(bundleDescriptor);
    when(bundleDescriptor.getVersion()).thenReturn(version);

    return plugin;
  }

  private Domain createMockDomain(String name, DomainStatus status) {
    Domain domain = createMockDeployableArtifact(Domain.class, name);

    // Mock descriptor
    DomainDescriptor descriptor = mock(DomainDescriptor.class);
    when(domain.getDescriptor()).thenReturn(descriptor);
    when(descriptor.getPlugins()).thenReturn(emptySet());

    // Set the domain status
    when(domain.getStatus()).thenReturn(status);

    // For DESTROYED status, domain might not have an artifact context (undeployed)
    if (status == DomainStatus.DESTROYED) {
      when(domain.getArtifactContext()).thenReturn(null);
    } else {
      ArtifactContext artifactContext = domain.getArtifactContext();

      // Mock MuleContext
      MuleContext muleContext = mock(MuleContext.class);
      when(artifactContext.getMuleContext()).thenReturn(muleContext);
    }

    return domain;
  }

  private <T extends DeployableArtifact<?>> T createMockDeployableArtifact(Class<T> artifactClass, String name) {
    T artifact = mock(artifactClass);
    when(artifact.getArtifactName()).thenReturn(name);

    // Mock artifact context and registry
    ArtifactContext artifactContext = mock(ArtifactContext.class);
    Registry registry = mock(Registry.class);
    when(artifact.getArtifactContext()).thenReturn(artifactContext);
    when(artifactContext.getRegistry()).thenReturn(registry);

    // Mock plugins - empty for basic test
    when(artifact.getArtifactPlugins()).thenReturn(emptyList());

    return artifact;
  }

  private Application createMockApplicationWithPolicies() throws Exception {
    Application app = createMockApplication("app-with-policies", STARTED);

    // Create a MuleApplicationPolicyProvider instance with two policies
    PolicyProvider policyProvider = createMuleApplicationPolicyProvider(app);

    // Setup registry to return the policy provider
    Registry registry = app.getArtifactContext().getRegistry();
    when(registry.lookupByType(PolicyProvider.class)).thenReturn(Optional.of(policyProvider));

    return app;
  }

  private PolicyProvider createMuleApplicationPolicyProvider(Application app) throws Exception {
    // Mock the factories
    PolicyTemplateFactory policyTemplateFactory = mock(PolicyTemplateFactory.class);
    PolicyInstanceProviderFactory policyInstanceProviderFactory = mock(PolicyInstanceProviderFactory.class);

    // Create a real instance of MuleApplicationPolicyProvider
    MuleApplicationPolicyProvider policyProvider =
        new MuleApplicationPolicyProvider(policyTemplateFactory, policyInstanceProviderFactory);
    policyProvider.setApplication(app);

    // Create and add first policy
    addPolicyToProvider(policyProvider, app, policyTemplateFactory, policyInstanceProviderFactory,
                        "testPolicy1", "1.0.0", "policy-1", 1);

    // Create and add second policy
    addPolicyToProvider(policyProvider, app, policyTemplateFactory, policyInstanceProviderFactory,
                        "testPolicy2", "2.0.0", "policy-2", 2);

    return policyProvider;
  }

  private void addPolicyToProvider(MuleApplicationPolicyProvider policyProvider, Application app,
                                   PolicyTemplateFactory policyTemplateFactory,
                                   PolicyInstanceProviderFactory policyInstanceProviderFactory,
                                   String artifactId, String version, String policyId, int order)
      throws Exception {
    PolicyTemplateDescriptor policyTemplateDescriptor = new PolicyTemplateDescriptor(artifactId);
    policyTemplateDescriptor.setBundleDescriptor(new Builder()
        .setArtifactId(artifactId)
        .setGroupId("test")
        .setVersion(version)
        .build());

    PolicyTemplate policyTemplate = createMockPolicyTemplate(artifactId, version);
    when(policyTemplateFactory.createArtifact(app, policyTemplateDescriptor)).thenReturn(policyTemplate);

    ApplicationPolicyInstance policyInstance = mock(ApplicationPolicyInstance.class);
    PolicyPointcut pointcut = mock(PolicyPointcut.class);
    when(policyInstance.getPolicyTemplate()).thenReturn(policyTemplate);
    when(policyInstance.getPointcut()).thenReturn(pointcut);
    when(policyInstance.getOrder()).thenReturn(order);

    PolicyParametrization parametrization =
        new PolicyParametrization(policyId, pointcut, order, emptyMap(), mock(File.class), emptyList());
    when(policyInstanceProviderFactory.create(app, policyTemplate, parametrization))
        .thenReturn(policyInstance);

    policyProvider.addPolicy(policyTemplateDescriptor, parametrization);
  }

  private PolicyTemplate createMockPolicyTemplate(String artifactId, String version) {
    PolicyTemplate policyTemplate = mock(PolicyTemplate.class);

    // Create mock plugins for the policy template
    ArtifactPlugin policyPlugin = createMockPlugin("PolicyPlugin", version);
    when(policyTemplate.getArtifactPlugins()).thenReturn(List.of(policyPlugin));

    // Mock descriptor
    PolicyTemplateDescriptor descriptor = new PolicyTemplateDescriptor("testPolicy");
    descriptor.setBundleDescriptor(new Builder()
        .setArtifactId(artifactId)
        .setGroupId("test")
        .setVersion(version)
        .build());
    when(policyTemplate.getDescriptor()).thenReturn(descriptor);

    // Mock getArtifactId() and getArtifactName() for template identification
    when(policyTemplate.getArtifactId()).thenReturn(artifactId);
    when(policyTemplate.getArtifactName()).thenReturn("testPolicy");

    // Mock ArtifactClassLoader (needed by addPolicy when removing templates)
    ArtifactClassLoader artifactClassLoader = mock(ArtifactClassLoader.class);
    when(policyTemplate.getArtifactClassLoader()).thenReturn(artifactClassLoader);

    return policyTemplate;
  }

  private Application createMockApplicationWithSpecificFeatureFlags(String... enabledFeatureNames) {
    Application app = createMockApplication("app-with-features", STARTED);

    // Create a FeatureFlaggingService that returns true for specific features
    FeatureFlaggingService featureFlaggingService = createMockFeatureFlaggingServiceWithSpecificFeatures(
                                                                                                         enabledFeatureNames);

    // Setup registry to return the feature flagging service
    Registry registry = app.getArtifactContext().getRegistry();
    when(registry.lookupByType(FeatureFlaggingService.class)).thenReturn(Optional.of(featureFlaggingService));

    return app;
  }

  private FeatureFlaggingService createMockFeatureFlaggingServiceWithSpecificFeatures(String... enabledFeatureNames) {
    FeatureFlaggingService featureFlaggingService = mock(FeatureFlaggingService.class);

    // Get all registered features from the registry
    Map<Feature, ?> registeredFeatures = getInstance().getFeatureFlagConfigurations();

    // Create a set of enabled feature names for quick lookup
    Set<String> enabledNames = of(enabledFeatureNames);

    // For each registered feature, check if its name matches one of the enabled feature names
    // If it does, mock it to return true; otherwise return false
    for (Feature feature : registeredFeatures.keySet()) {
      boolean isEnabled = enabledNames.contains(feature.name());
      when(featureFlaggingService.isEnabled(feature)).thenReturn(isEnabled);
    }

    return featureFlaggingService;
  }
}

