/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.test.module.tls;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.arrayContaining;
import static org.hamcrest.Matchers.containsString;
import static org.hamcrest.Matchers.endsWith;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.sameInstance;

import static org.junit.jupiter.api.Assertions.assertThrows;

import static org.mockito.Mockito.mock;

import org.mule.runtime.api.lifecycle.CreateException;
import org.mule.runtime.api.tls.TlsContextFactory;
import org.mule.runtime.api.tls.TlsContextFactoryBuilder;
import org.mule.runtime.api.tls.TlsContextKeyStoreConfiguration;
import org.mule.runtime.api.tls.TlsContextTrustStoreConfiguration;
import org.mule.runtime.module.tls.internal.DefaultTlsContextFactoryBuilder;
import org.mule.tck.junit4.AbstractMuleTestCase;

import java.security.KeyStore;

import javax.net.ssl.TrustManagerFactory;

import org.junit.jupiter.api.Test;

public class DefaultTlsContextFactoryBuilderTestCase extends AbstractMuleTestCase {

  private final TlsContextFactory defaultFactoryMock = mock(TlsContextFactory.class);
  private final TlsContextFactoryBuilder builder = new DefaultTlsContextFactoryBuilder(defaultFactoryMock);

  @Test
  void buildsContext() throws Exception {
    TlsContextFactory contextFactory = builder
        .enabledCipherSuites("TLS_SOMETHING")
        .enabledProtocols("TLSv1.1")
        .keyStorePath("serverKeystore")
        .keyStorePassword("mulepassword")
        .keyAlias("muleserver")
        .keyPassword("mulepassword")
        .keyStoreAlgorithm("PKIX")
        .trustStorePath("trustStore")
        .trustStorePassword("mulepassword")
        .trustStoreType("jceks")
        .insecureTrustStore(true)
        .build();

    assertThat(contextFactory.getEnabledProtocols(), is(arrayContaining("TLSv1.1")));
    assertThat(contextFactory.getEnabledCipherSuites(), is(arrayContaining("TLS_SOMETHING")));

    TlsContextKeyStoreConfiguration keyStoreConfiguration = contextFactory.getKeyStoreConfiguration();
    assertThat(keyStoreConfiguration.getPath(), endsWith("serverKeystore"));
    assertThat(keyStoreConfiguration.getPassword(), is("mulepassword"));
    assertThat(keyStoreConfiguration.getKeyPassword(), is("mulepassword"));
    assertThat(keyStoreConfiguration.getAlias(), is("muleserver"));
    assertThat(keyStoreConfiguration.getType(), is(KeyStore.getDefaultType()));
    assertThat(keyStoreConfiguration.getAlgorithm(), is("PKIX"));

    TlsContextTrustStoreConfiguration trustStoreConfiguration = contextFactory.getTrustStoreConfiguration();
    assertThat(trustStoreConfiguration.getPath(), endsWith("trustStore"));
    assertThat(trustStoreConfiguration.getPassword(), is("mulepassword"));
    assertThat(trustStoreConfiguration.getType(), is("jceks"));
    assertThat(trustStoreConfiguration.getAlgorithm(), is(TrustManagerFactory.getDefaultAlgorithm()));
  }

  @Test
  void returnsDefaultContext() {
    assertThat(defaultFactoryMock, is(sameInstance(builder.buildDefault())));
  }

  @Test
  void failsWhenFileNotFound() throws Exception {
    var thrown = assertThrows(CreateException.class, () -> builder.trustStorePath("aPath").build());
    assertThat(thrown.getMessage(), containsString("Resource aPath could not be found"));
  }

  @Test
  void failsWhenStorePasswordIsWrong() throws Exception {
    var thrown =
        assertThrows(CreateException.class,
                     () -> builder.keyStorePath("serverKeystore").keyStorePassword("zaraza").keyPassword("mulepassword").build());
    assertThat(thrown.getMessage(), containsString("Unable to initialise TLS configuration"));
  }

  @Test
  void failsWhenKeyPasswordIsWrong() throws Exception {
    var thrown =
        assertThrows(CreateException.class,
                     () -> builder.keyStorePath("serverKeystore").keyStorePassword("mulepassword").keyPassword("zaraza").build());
    assertThat(thrown.getMessage(), containsString("Unable to initialise TLS configuration"));
  }

}
