/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.service.internal.test.manager;

import static java.util.Optional.empty;

import static org.hamcrest.CoreMatchers.containsString;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.sameInstance;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.core.IsNull.notNullValue;

import static org.junit.Assert.assertThrows;

import org.mule.runtime.api.service.Service;
import org.mule.runtime.api.service.ServiceDefinition;
import org.mule.runtime.api.service.ServiceProvider;
import org.mule.runtime.module.service.api.discoverer.ServiceResolutionError;
import org.mule.runtime.module.service.internal.manager.DefaultServiceRegistry;
import org.mule.tck.junit4.AbstractMuleTestCase;

import java.util.Optional;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;

import jakarta.inject.Inject;

@ExtendWith(MockitoExtension.class)
public class ServiceRegistryTestCase extends AbstractMuleTestCase {

  private DefaultServiceRegistry serviceRegistry;
  private ServiceA serviceA;
  @Mock
  private ServiceProvider serviceProviderA;

  @BeforeEach
  public void before() {
    serviceRegistry = new DefaultServiceRegistry();
    serviceA = new ServiceA();
  }

  private void registerServiceA() {
    serviceRegistry.register(serviceA, ServiceA.class);
  }

  @Test
  void injectRequiredServiceDependency() throws ServiceResolutionError {
    registerServiceA();

    ServiceProviderRequiredDependencyToServiceA serviceProviderRequiredDependencyToServiceA =
        new ServiceProviderRequiredDependencyToServiceA();
    serviceRegistry.inject(serviceProviderRequiredDependencyToServiceA);

    assertThat(serviceProviderRequiredDependencyToServiceA.serviceA, is(notNullValue()));
    assertThat(serviceProviderRequiredDependencyToServiceA.serviceA, sameInstance(serviceA));
  }

  @Test
  void injectRequiredServiceDependencyJavax() throws ServiceResolutionError {
    registerServiceA();

    ServiceProviderRequiredJavaxDependencyToServiceA serviceProviderRequiredDependencyToServiceA =
        new ServiceProviderRequiredJavaxDependencyToServiceA();
    serviceRegistry.inject(serviceProviderRequiredDependencyToServiceA);

    assertThat(serviceProviderRequiredDependencyToServiceA.serviceA, is(notNullValue()));
    assertThat(serviceProviderRequiredDependencyToServiceA.serviceA, sameInstance(serviceA));
  }

  @Test
  void missingRequiredServiceDependency() throws ServiceResolutionError {
    ServiceProviderRequiredDependencyToServiceA serviceProviderRequiredDependencyToServiceA =
        new ServiceProviderRequiredDependencyToServiceA();

    var thrown =
        assertThrows(ServiceResolutionError.class, () -> serviceRegistry.inject(serviceProviderRequiredDependencyToServiceA));
    assertThat(thrown.getMessage(), containsString("Cannot find a service to inject into field "
        + "'" + ServiceProviderRequiredDependencyToServiceA.class.getName() + "#serviceA' of type '"
        + ServiceA.class.getName() + "'"));
  }

  @Test
  void injectEmptyOptionalServiceDependency() throws ServiceResolutionError {
    ServiceProviderOptionalDependencyToServiceA serviceProviderOptionalDependencyToServiceA =
        new ServiceProviderOptionalDependencyToServiceA();
    serviceRegistry.inject(serviceProviderOptionalDependencyToServiceA);

    assertThat(serviceProviderOptionalDependencyToServiceA.serviceA, is(empty()));
  }

  @Test
  void injectOptionalServiceDependency() throws ServiceResolutionError {
    registerServiceA();

    ServiceProviderOptionalDependencyToServiceA serviceProviderOptionalDependencyToServiceA =
        new ServiceProviderOptionalDependencyToServiceA();
    serviceRegistry.inject(serviceProviderOptionalDependencyToServiceA);

    assertThat(serviceProviderOptionalDependencyToServiceA.serviceA.isPresent(), is(true));
    assertThat(serviceProviderOptionalDependencyToServiceA.serviceA.get(), sameInstance(serviceA));
  }

  @Test
  void injectOptionalSetterServiceDependency() throws ServiceResolutionError {
    registerServiceA();

    ServiceProviderOptionalSetterDependencyToServiceA serviceProviderOptionalSetterDependencyToServiceA =
        new ServiceProviderOptionalSetterDependencyToServiceA();
    serviceRegistry.inject(serviceProviderOptionalSetterDependencyToServiceA);

    assertThat(serviceProviderOptionalSetterDependencyToServiceA.serviceA.isPresent(), is(true));
    assertThat(serviceProviderOptionalSetterDependencyToServiceA.serviceA.get(), sameInstance(serviceA));
  }

  @Test
  void injectSetterServiceDependency() throws ServiceResolutionError {
    registerServiceA();

    ServiceProviderSetterDependencyToServiceA serviceProviderOptionalDependencyToServiceA =
        new ServiceProviderSetterDependencyToServiceA();
    serviceRegistry.inject(serviceProviderOptionalDependencyToServiceA);

    assertThat(serviceProviderOptionalDependencyToServiceA.serviceA, sameInstance(serviceA));
  }

  @Test
  void injectFailingSetterServiceDependency() throws ServiceResolutionError {
    registerServiceA();

    ServiceProviderFailingSetterDependencyToServiceA serviceProviderOptionalDependencyToServiceA =
        new ServiceProviderFailingSetterDependencyToServiceA();

    assertThrows(ServiceResolutionError.class, () -> serviceRegistry.inject(serviceProviderOptionalDependencyToServiceA));
  }

  public class ServiceA implements Service {

    @Override
    public String getName() {
      return this.getClass().getSimpleName();
    }
  }

  public class ServiceProviderRequiredJavaxDependencyToServiceA implements ServiceProvider {

    @javax.inject.Inject
    ServiceA serviceA;

    @Override
    public ServiceDefinition getServiceDefinition() {
      return null;
    }
  }

  public class ServiceProviderRequiredDependencyToServiceA implements ServiceProvider {

    @Inject
    ServiceA serviceA;

    @Override
    public ServiceDefinition getServiceDefinition() {
      return null;
    }
  }

  public class ServiceProviderOptionalDependencyToServiceA implements ServiceProvider {

    @Inject
    Optional<ServiceA> serviceA;

    @Override
    public ServiceDefinition getServiceDefinition() {
      return null;
    }
  }

  public class ServiceProviderOptionalSetterDependencyToServiceA implements ServiceProvider {

    Optional<ServiceA> serviceA;

    @Override
    public ServiceDefinition getServiceDefinition() {
      return null;
    }

    @Inject
    public void setServiceA(Optional<ServiceA> serviceA) {
      this.serviceA = serviceA;
    }

  }

  public class ServiceProviderSetterDependencyToServiceA implements ServiceProvider {

    ServiceA serviceA;

    @Override
    public ServiceDefinition getServiceDefinition() {
      return null;
    }

    @Inject
    public void setServiceA(ServiceA serviceA) {
      this.serviceA = serviceA;
    }
  }

  public class ServiceProviderFailingSetterDependencyToServiceA implements ServiceProvider {

    ServiceA serviceA;

    @Override
    public ServiceDefinition getServiceDefinition() {
      return null;
    }

    @Inject
    public void setServiceA(ServiceA serviceA) {
      throw new NullPointerException("Expected");
    }
  }

}
