/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.service.internal.manager;

import static java.lang.String.format;
import static java.util.Optional.ofNullable;

import static org.reflections.ReflectionUtils.getAllFields;
import static org.reflections.ReflectionUtils.getAllMethods;
import static org.reflections.util.ReflectionUtilsPredicates.withAnnotation;

import org.mule.runtime.api.service.Service;
import org.mule.runtime.api.service.ServiceProvider;
import org.mule.runtime.module.service.api.discoverer.ServiceResolutionError;
import org.mule.runtime.module.service.api.manager.ServiceRegistry;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.util.Collection;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Optional;
import java.util.function.Supplier;

import jakarta.inject.Inject;
import jakarta.inject.Named;
import jakarta.inject.Qualifier;

/**
 * Keeps track of {@link Service} implementations and is capable of injecting them into {@link ServiceProvider} instances with
 * fields annotated with {@link Inject}. Optionality is supported by using {@link Optional} type when declared a field with
 * {@link Inject}. Method injection as {@link Qualifier} and {@link Named} annotations are not supported.
 *
 * @since 4.2
 */
public class DefaultServiceRegistry implements ServiceRegistry {

  @FunctionalInterface
  private interface PropertyInjector {

    void inject(Object dependency) throws IllegalArgumentException, IllegalAccessException, InvocationTargetException;

  }

  private final Map<Class<? extends Service>, Service> services = new HashMap<>();

  /**
   * Injects the tracked {@link Service services} into the given {@code serviceProvider}
   *
   * @param serviceProvider the injection target
   * @throws ServiceResolutionError if a dependency could not be injected
   */
  public void inject(ServiceProvider serviceProvider) throws ServiceResolutionError {
    doInject(serviceProvider, Inject.class);
    doInject(serviceProvider, javax.inject.Inject.class);
  }

  private void doInject(ServiceProvider serviceProvider, final Class<? extends Annotation> injectAnnotation)
      throws ServiceResolutionError {
    for (Field field : getAllFields(serviceProvider.getClass(), withAnnotation(injectAnnotation))) {
      injectProperty(field.getType(),
                     field::getGenericType,
                     dependency -> {
                       field.setAccessible(true);
                       field.set(serviceProvider, dependency);
                     },
                     format("into field '%s#%s'",
                            field.getDeclaringClass().getName(),
                            field.getName()));
    }
    for (Method method : getAllMethods(serviceProvider.getClass(), withAnnotation(injectAnnotation))) {
      if (method.getParameters().length != 1) {
        continue;
      }

      injectProperty(method.getParameterTypes()[0],
                     () -> method.getGenericParameterTypes()[0],
                     dependency -> method.invoke(serviceProvider, dependency),
                     format("on method '%s#%s'",
                            method.getDeclaringClass().getName(),
                            method.getName()));
    }
  }

  private void injectProperty(Class<?> dependencyType, Supplier<Type> genericTypeSupplier,
                              PropertyInjector propertyInjector, String injectionTargetDescription)
      throws ServiceResolutionError {
    Object dependency = resolveObjectToInject(dependencyType, genericTypeSupplier,
                                              injectionTargetDescription);
    try {
      propertyInjector.inject(dependency);
    } catch (Exception e) {
      throw new ServiceResolutionError(format("Could not inject dependency %s of type '%s'",
                                              injectionTargetDescription,
                                              dependencyType.getName()),
                                       e);
    }
  }

  private Object resolveObjectToInject(Class<?> dependencyType, Supplier<Type> genericType, String injectionTarget)
      throws ServiceResolutionError {
    boolean asOptional = false;
    if (dependencyType.equals(Optional.class)) {
      dependencyType = (Class<?>) ((ParameterizedType) genericType.get()).getActualTypeArguments()[0];
      asOptional = true;
    }

    Object dependency = doResolveObjectToInject(dependencyType);

    if (dependency == null && !asOptional) {
      throw new ServiceResolutionError(format("Cannot find a service to inject %s of type '%s'",
                                              injectionTarget,
                                              dependencyType.getName()));
    }

    return asOptional ? ofNullable(dependency) : dependency;
  }

  private Object doResolveObjectToInject(Class<?> dependencyType) {
    return services.entrySet().stream()
        .filter(entry -> dependencyType.isAssignableFrom(entry.getKey()))
        .findFirst()
        .map(Entry::getValue)
        .orElse(null);
  }

  @Override
  public <S extends Service> void register(S service, Class<? extends S> serviceContract) {
    services.put(serviceContract, service);
  }

  public <S extends Service> void register(Class<? extends S> serviceContract, S service) {
    services.put(serviceContract, service);
  }

  @Override
  public <S extends Service> void unregister(Class<? extends S> serviceContract) {
    services.remove(serviceContract);
  }

  @Override
  public <S extends Service> Optional<S> getService(Class<? extends S> serviceInterface) {
    return services
        .values()
        .stream()
        .filter(serviceInterface::isInstance)
        .map(s -> (S) s)
        .findAny();
  }

  @Override
  public Collection<Service> getAllServices() {
    return new HashSet<>(services.values());
  }
}
