/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.observability;

import static org.mule.runtime.api.util.MuleSystemProperties.ENABLE_OBSERVABILITY_CONFIGURATION_AT_APPLICATION_LEVEL_PROPERTY;
import static org.mule.runtime.api.util.MuleSystemProperties.ENABLE_TRACER_CONFIGURATION_AT_APPLICATION_LEVEL_PROPERTY;

import static java.lang.Boolean.getBoolean;
import static java.lang.String.format;
import static java.lang.System.getProperty;
import static java.nio.file.Path.of;
import static java.util.Optional.empty;

import org.mule.runtime.config.api.properties.ConfigurationPropertiesResolver;
import org.mule.runtime.config.internal.model.dsl.config.DefaultConfigurationPropertiesResolver;
import org.mule.runtime.config.internal.model.dsl.config.MapConfigurationPropertiesProvider;
import org.mule.runtime.config.internal.model.dsl.config.SystemPropertiesConfigurationProvider;
import org.mule.runtime.module.artifact.api.descriptor.DeployableArtifactDescriptor;
import org.mule.runtime.module.observability.configuration.ObservabilitySignalConfiguration;
import org.mule.runtime.module.observability.configuration.ObservabilitySignalConfigurationFileFinder;
import org.mule.runtime.module.observability.configuration.ObservabilitySignalConfigurationPropertyResolver;

import java.io.File;
import java.io.IOException;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Properties;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.dataformat.yaml.YAMLFactory;

/**
 * A class that provides functionality to read configuration from a file.
 *
 * @since 4.6.0
 */
public abstract class AbstractFileObservabilitySignalConfiguration implements ObservabilitySignalConfiguration {

  private static final ObjectMapper configFileMapper = new ObjectMapper(new YAMLFactory());
  private static final String UNSUPPORTED_TYPE_ERROR_MESSAGE =
      "Configuration key '%s' has an unsupported type %s. Only string values are allowed.";
  private static final String OPEN_TELEMETRY_ARTIFACT_LEVEL_CONFIGURATION_PROPERTIES =
      "OpenTelemetry artifact level configuration properties. Artifact name: %s.";
  // This one cannot be static because the system properties are only read at creation by the
  // SystemPropertiesConfigurationProvider.
  private final ConfigurationPropertiesResolver systemPropertiesOnlyConfigurationPropertiesResolver =
      new DefaultConfigurationPropertiesResolver(empty(), new SystemPropertiesConfigurationProvider());
  private final ObservabilitySignalConfigurationPropertyResolver observabilitySignalRuntimeLevelConfigurationPropertyResolver =
      systemPropertiesOnlyConfigurationPropertiesResolver::apply;
  private final ObservabilitySignalConfigurationPropertyResolver observabilitySignalArtifactLevelConfigurationPropertyResolver;
  private boolean isPropertiesInitialized = false;
  private JsonNode configuration;
  private File configurationFile;
  private final String artifactId;
  private final ObservabilitySignalConfigurationFileFinder artifactResourceFinder;

  protected AbstractFileObservabilitySignalConfiguration(ObservabilitySignalConfigurationFileFinder artifactResourceFinder,
                                                         DeployableArtifactDescriptor deployableArtifactDescriptor) {
    this(artifactResourceFinder, getArtifactDeploymentProperties(deployableArtifactDescriptor),
         getArtifactId(deployableArtifactDescriptor));
  }

  protected AbstractFileObservabilitySignalConfiguration(ObservabilitySignalConfigurationFileFinder artifactResourceFinder,
                                                         Properties artifactDeploymentProperties,
                                                         String artifactId) {
    this.artifactId = artifactId;
    this.observabilitySignalArtifactLevelConfigurationPropertyResolver =
        getArtifactLevelConfigurationPropertyResolver(artifactDeploymentProperties, artifactId);
    this.artifactResourceFinder = artifactResourceFinder;
  }

  @Override
  public String getStringValue(String configurationKey) {
    initialiseIfNecessary();
    String configurationValue = readStringFromConfigOrSystemProperty(configurationKey);
    return resolveConfigurationProperty(configurationKey, configurationValue);
  }

  @Override
  public List<String> getStringListValue(String configurationKey) {
    initialiseIfNecessary();
    if (configuration == null) {
      return null;
    }
    return readStringListFromConfig(configurationKey);
  }

  @Override
  public Path getPathValue(String key, Path defaultValue) {
    return getAbsolutePath(ObservabilitySignalConfiguration.super.getPathValue(key, defaultValue));
  }


  /**
   * Invoked when the configuration file is not found.
   */
  protected abstract void onConfigurationFileNotFound();

  /**
   * Invoked when an error during configuration file loading occurs.
   *
   * @param error             The error.
   * @param configurationFile The failed configuration file.
   */
  protected abstract void onConfigurationFileLoadError(Exception error, File configurationFile);

  /**
   * @return The name of the main signal configuration file (additional files can be referenced from this configuration, such as a
   *         TLS certificate).
   */
  protected abstract String getSignalConfigurationFileName();

  /**
   * @return Absolute path to the signal configuration files directory (does not include a file name).
   */
  protected abstract Path getSignalConfigurationFileDirectoryPath();

  protected static File findArtifactConfigFile(ClassLoader executionClassloader, String configFilePath) {
    try {
      URL resource = executionClassloader.getResource(configFilePath);
      return resource != null ? new File(resource.toURI()) : null;
    } catch (URISyntaxException e) {
      return null;
    }
  }

  protected File getConfigurationFile() {
    return configurationFile;
  }

  protected String getConfigurationFileLocations() {
    if (isApplicationLevelConfigurable()) {
      return format("at at both '%s' artifact resources and the '%s' configuration path",
                    artifactId, getSignalConfigurationFileDirectoryPath());
    } else {
      return format("at the '%s' configuration path", getSignalConfigurationFileDirectoryPath());
    }
  }

  /**
   * @return True if the configuration file can be part of a mule application resources.
   */
  protected static boolean isApplicationLevelConfigurable() {
    return getBoolean(ENABLE_OBSERVABILITY_CONFIGURATION_AT_APPLICATION_LEVEL_PROPERTY)
        || getBoolean(ENABLE_TRACER_CONFIGURATION_AT_APPLICATION_LEVEL_PROPERTY);
  }

  /**
   * Not all the configuration values can be resolved at an artifact level, where different artifacts can have different
   * configuration values. This method defines if a configuration value can be resolved at an artifact level or not.
   *
   * @param configurationKey A configuration key.
   *
   * @return true if the configuration value can be resolved at an artifact level.
   */
  protected abstract boolean isArtifactLevelProperty(String configurationKey);

  protected void initialise() {
    loadConfiguration();
  }

  private static String getArtifactId(DeployableArtifactDescriptor deployableArtifactDescriptor) {
    return deployableArtifactDescriptor != null ? deployableArtifactDescriptor.getName() : "mule-runtime";
  }

  private static Properties getArtifactDeploymentProperties(DeployableArtifactDescriptor deployableArtifactDescriptor) {
    return deployableArtifactDescriptor != null ? deployableArtifactDescriptor.getDeploymentProperties().orElse(new Properties())
        : new Properties();
  }

  /**
   * @return The property resolver for the artifact level configuration values, which uses a Deployment properties / System
   *         properties hierarchy.
   */
  private static ObservabilitySignalConfigurationPropertyResolver getArtifactLevelConfigurationPropertyResolver(Properties configurationProperties,
                                                                                                                String artifactId) {
    return new ObservabilitySignalConfigurationPropertyResolver() {

      final ConfigurationPropertiesResolver configurationPropertiesResolver =
          new DefaultConfigurationPropertiesResolver(Optional.of(new DefaultConfigurationPropertiesResolver(empty(),
                                                                                                            new SystemPropertiesConfigurationProvider())),
                                                     new MapConfigurationPropertiesProvider(filterStringConfigurationProperties(configurationProperties),
                                                                                            String
                                                                                                .format(OPEN_TELEMETRY_ARTIFACT_LEVEL_CONFIGURATION_PROPERTIES,
                                                                                                        artifactId)));

      @Override
      public String resolve(String propertyReference) {
        return configurationPropertiesResolver.apply(propertyReference);
      }

    };
  }

  /**
   * Filter the String -> String properties from a Java properties object.
   *
   * @param configurationProperties The Java properties object.
   * @return Map of String -> String properties.
   */
  private static Map<String, String> filterStringConfigurationProperties(Properties configurationProperties) {
    Map<String, String> stringConfigurationProperties = new HashMap<>(configurationProperties.stringPropertyNames().size());
    for (String key : configurationProperties.stringPropertyNames()) {
      stringConfigurationProperties.put(key, configurationProperties.getProperty(key));
    }
    return stringConfigurationProperties;
  }

  /**
   * Resolves a configuration value if it is a configuration property reference.
   *
   * @param configurationKey   The configuration property key.
   * @param configurationValue The configuration property value, which can be a configuration property reference.
   * @return The resolved value or the original value if it is not a configuration property reference.
   */
  private String resolveConfigurationProperty(String configurationKey, String configurationValue) {
    if (isArtifactLevelProperty(configurationKey)) {
      return observabilitySignalArtifactLevelConfigurationPropertyResolver.resolve(configurationValue);
    } else {
      return observabilitySignalRuntimeLevelConfigurationPropertyResolver.resolve(configurationValue);
    }
  }

  private void initialiseIfNecessary() {
    if (!isPropertiesInitialized) {
      isPropertiesInitialized = true;
      initialise();
    }
  }

  private void loadConfiguration() {
    try {
      configurationFile = getSignalConfigurationFromArtifactOrFromFileSystem();
      if (configurationFile.exists()) {
        parseConfiguration(configurationFile);
      } else {
        onConfigurationFileNotFound();
      }
    } catch (Exception configurationException) {
      onConfigurationFileLoadError(configurationException, configurationFile);
    }
  }

  private File getSignalConfigurationFromArtifactOrFromFileSystem() {
    return getSignalConfigurationResourceFromArtifactOrFromFileSystem(of(getSignalConfigurationFileName()));
  }

  private File getSignalConfigurationResourceFromArtifactOrFromFileSystem(Path path) {
    File configurationFile = null;
    if (isApplicationLevelConfigurable()) {
      // This delegates into artifact level signal configuration. Artifact signal configuration must be at the root of the
      // artifact resources.
      configurationFile = artifactResourceFinder.getResource(path.toString());
    }
    // This searches based on the directory of the main signal configuration file when there is not artifact level configuration.
    return configurationFile != null ? configurationFile : getSignalConfigurationFileDirectoryPath().resolve(path).toFile();
  }

  private String readStringFromConfigOrSystemProperty(String key) {
    // If the configuration is not initialized, return the system property value.
    if (configuration == null) {
      return getProperty(key);
    }

    JsonNode node = readFromConfiguration(key, configuration);
    if (node == null || node.isNull()) {
      return null;
    }
    if (!node.isValueNode()) {
      throw new IllegalArgumentException(format(UNSUPPORTED_TYPE_ERROR_MESSAGE, key, node.getNodeType()));
    }
    String value = node.asText();
    return !value.isEmpty() ? value : null;
  }

  protected List<String> readStringListFromConfig(String key) {
    JsonNode node = readFromConfiguration(key, configuration);
    if (node == null || node.isNull()) {
      return null;
    }

    List<String> result = new ArrayList<>();
    if (!node.isArray()) {
      throw new IllegalArgumentException(format(UNSUPPORTED_TYPE_ERROR_MESSAGE, key, node.getNodeType()));
    }

    for (JsonNode elem : node) {
      if (!elem.isValueNode()) {
        throw new IllegalArgumentException(format(UNSUPPORTED_TYPE_ERROR_MESSAGE, key, elem.getNodeType()));
      }
      final String raw = elem.asText();
      if (raw != null && !raw.trim().isEmpty()) {
        String resolved = resolveConfigurationProperty(key, raw);
        if (!resolved.isEmpty()) {
          result.add(resolved);
        }
      }
    }
    return result;
  }

  private static JsonNode readFromConfiguration(String key, JsonNode configuration) {
    String[] path = key.split("\\.");
    JsonNode node = configuration;
    for (String pathPart : path) {
      node = node.get(pathPart);
      if (node == null) {
        return null;
      }
    }
    return node;
  }

  private Path getAbsolutePath(Path path) {
    if (path.isAbsolute()) {
      return path;
    }
    // We need to resolve the non-absolute path because it can be either at the artifact resources via
    // ENABLE_OBSERVABILITY_CONFIGURATION_AT_APPLICATION_LEVEL_PROPERTY or at the file system.
    File absolutePathConversionFile = getSignalConfigurationResourceFromArtifactOrFromFileSystem(path);
    return absolutePathConversionFile.toPath().toAbsolutePath();
  }

  private void parseConfiguration(File configurationFile) throws IOException {
    configuration = configFileMapper.readTree(configurationFile);
  }
}
