/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.observability.configuration;

import static org.junit.jupiter.api.Assertions.assertEquals;
import static org.junit.jupiter.api.Assertions.assertThrows;
import static org.mule.runtime.module.observability.configuration.HeadersParser.parseHeaders;

import java.util.Map;

import org.junit.jupiter.api.Test;

/**
 * Test case for {@link HeadersParser}.
 */
public class HeadersParserTestCase {

  // ========================================
  // SINGLE VALUE TESTS
  // ========================================

  @Test
  public void parseSingleHeaderWithValue() {
    Map<String, String> result = parseHeaders("header1=value1");

    assertEquals(1, result.size());
    assertEquals("value1", result.get("header1"));
  }

  @Test
  public void parseSingleHeaderWithValueWithSpaces() {
    Map<String, String> result = parseHeaders("header1=value 1 with spaces");

    assertEquals(1, result.size());
    assertEquals("value 1 with spaces", result.get("header1"));
  }

  @Test
  public void parseSingleHeaderWithoutValue() {
    Map<String, String> result = parseHeaders("auth-header");

    assertEquals(1, result.size());
    assertEquals("", result.get("auth-header"));
  }

  @Test
  public void parseSingleHeaderWithValueContainingEquals() {
    Map<String, String> result = parseHeaders("header1=value=with=equals");

    assertEquals(1, result.size());
    assertEquals("value=with=equals", result.get("header1"));
  }

  @Test
  public void parseSingleHeaderWithEmptyValue() {
    // Empty value with '=' should fail
    assertThrows(IllegalArgumentException.class, () -> parseHeaders("header1="));
  }

  @Test
  public void parseSingleHeaderWithEmptyName() {
    // Empty header name should fail
    assertThrows(IllegalArgumentException.class, () -> parseHeaders("=value1"));
  }

  @Test
  public void parseSingleHeaderWithBothEmpty() {
    // Both name and value empty should fail
    assertThrows(IllegalArgumentException.class, () -> parseHeaders("="));
  }

  // ========================================
  // COMMA-SEPARATED VALUES TESTS
  // ========================================

  @Test
  public void parseCommaSeparatedHeadersWithValues() {
    Map<String, String> result = parseHeaders("header1=value1,header2=value2");

    assertEquals(2, result.size());
    assertEquals("value1", result.get("header1"));
    assertEquals("value2", result.get("header2"));
  }

  @Test
  public void parseCommaSeparatedHeadersWithWhitespace() {
    // Tests whitespace: leading/trailing, around '=', and after commas
    Map<String, String> result = parseHeaders(" header1 = value1 , header2=value2, header3 = value3 ");

    assertEquals(3, result.size());
    assertEquals("value1", result.get("header1"));
    assertEquals("value2", result.get("header2"));
    assertEquals("value3", result.get("header3"));
  }

  @Test
  public void parseCommaSeparatedHeadersWithoutValues() {
    Map<String, String> result = parseHeaders("header1,header2,header3");

    assertEquals(3, result.size());
    assertEquals("", result.get("header1"));
    assertEquals("", result.get("header2"));
    assertEquals("", result.get("header3"));
  }

  @Test
  public void parseCommaSeparatedMixedHeadersWithAndWithoutValues() {
    Map<String, String> result = parseHeaders("header1=value1,header2,header3=value3");

    assertEquals(3, result.size());
    assertEquals("value1", result.get("header1"));
    assertEquals("", result.get("header2"));
    assertEquals("value3", result.get("header3"));
  }

  @Test
  public void parseCommaSeparatedHeadersWithEmptyEntries() {
    // Empty entries should cause an exception
    assertThrows(IllegalArgumentException.class, () -> parseHeaders("header1=value1,,header2=value2"));
  }

  // ========================================
  // EDGE CASES - INVALID INPUTS
  // ========================================

  @Test
  public void parseNullThrowsException() {
    assertThrows(IllegalArgumentException.class, () -> parseHeaders(null));
  }

  @Test
  public void parseEmptyStringThrowsException() {
    // Empty string should cause an exception
    assertThrows(IllegalArgumentException.class, () -> parseHeaders(""));
  }

  @Test
  public void parseStringWithOnlySpacesThrowsException() {
    // String with only spaces should cause an exception
    assertThrows(IllegalArgumentException.class, () -> parseHeaders("   "));
  }
}
