/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.observability.configuration;

import java.util.HashMap;
import java.util.Map;

/**
 * Utility class for parsing headers configuration in comma-separated format.
 * <p>
 * Supports the following formats:
 * <ul>
 * <li>Comma-separated headers with values: {@code header1=value1,header2=value2}</li>
 * <li>Headers without values: {@code header1,header2}</li>
 * <li>Mixed format: {@code header1=value1,header2}</li>
 * </ul>
 * <p>
 * Examples:
 *
 * <pre>
 *
 * Map&lt;String, String&gt; headers = HeadersParser.parseHeaders("auth-token=abc123,user-agent=mule");
 * // Result: {"auth-token": "abc123", "user-agent": "mule"}
 *
 * Map&lt;String, String&gt; headers = HeadersParser.parseHeaders("authorization");
 * // Result: {"authorization": ""}
 * </pre>
 *
 * @since 4.11.0
 */
public final class HeadersParser {

  private HeadersParser() {
    // Utility class
  }

  /**
   * Parses a comma-separated string of headers into a Map.
   * <p>
   * Each entry can be in the format:
   * <ul>
   * <li>{@code headerName=headerValue} - header with value</li>
   * <li>{@code headerName} - header without value (empty string)</li>
   * </ul>
   * <p>
   * Empty entries are not allowed and will cause an exception. Leading and trailing whitespace is trimmed from both names and
   * values.
   *
   * @param headers comma-separated string of headers (e.g., "h1=v1,h2=v2")
   * @return Map of header names to values (never null, may be empty)
   * @throws IllegalArgumentException if headers string is null, empty, or contains invalid entries
   */
  public static Map<String, String> parseHeaders(String headers) {
    if (headers == null) {
      throw new IllegalArgumentException("Headers string cannot be null");
    }
    if (headers.trim().isEmpty()) {
      throw new IllegalArgumentException("Headers configuration has an empty value");
    }

    Map<String, String> result = new HashMap<>();
    String[] entries = headers.split(",");

    for (String entry : entries) {
      String trimmedEntry = entry.trim();
      if (trimmedEntry.isEmpty()) {
        throw new IllegalArgumentException("Headers configuration contain empty entries");
      }

      int separatorIndex = trimmedEntry.indexOf("=");
      if (separatorIndex == -1) {
        // Header without value: "header1"
        result.put(trimmedEntry, "");
      } else {
        // Header with value: "header1=value1"
        String headerName = trimmedEntry.substring(0, separatorIndex).trim();
        if (headerName.isEmpty()) {
          throw new IllegalArgumentException("Headers configuration contain empty header names");
        }
        String headerValue = trimmedEntry.substring(separatorIndex + 1).trim();
        if (headerValue.isEmpty()) {
          throw new IllegalArgumentException("Headers configuration contain empty header values");
        }
        result.put(headerName, headerValue);
      }
    }

    return result;
  }
}

