/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.observability.configuration;

import static java.lang.Boolean.parseBoolean;
import static java.lang.Double.parseDouble;
import static java.lang.Integer.parseInt;
import static java.lang.Long.parseLong;
import static java.nio.file.Path.of;
import static java.time.Duration.ofSeconds;

import java.nio.file.Path;
import java.time.Duration;

/**
 * Configuration for the export of an observability signal (tracing, metrics, logging).
 *
 * @since 4.10.0
 */
public interface ObservabilitySignalConfiguration {

  /**
   * Returns the String value of a configuration parameter.
   *
   * @param key the key of a configuration parameter.
   * @return the String value associated to the {@param key} or null if not found.
   */
  String getStringValue(String key);

  /**
   * Returns the String value of a configuration parameter or the default value if no configured value is found.
   *
   * @param key the key of a configuration parameter.
   * @return the String value associated to the {@param key} or the default value if the value is not found.
   */
  default String getStringValue(String key, String defaultValue) {
    String value = getStringValue(key);
    if (value != null) {
      return value;
    }
    return defaultValue;
  }

  /**
   * Returns the Path value of a configuration parameter.
   *
   * @param key the key of a configuration parameter.
   * @return the Path value associated to the {@param key} or null if not found.
   */
  default Path getPathValue(String key) {
    return getPathValue(key, null);
  }

  /**
   * Returns the Path value of a configuration parameter or the default value if no configured value is found.
   *
   * @param key the key of a configuration parameter.
   * @return the Path value associated to the {@param key} or the default value if the value is not found.
   */
  default Path getPathValue(String key, Path defaultValue) {
    String value = getStringValue(key);
    if (value != null) {
      return of(value);
    }
    return defaultValue;
  }

  /**
   * Returns the Boolean value of a configuration parameter.
   *
   * @param key the key of a configuration parameter.
   * @return the Boolean value associated to the {@param key} or null if not found.
   */
  default Boolean getBooleanValue(String key) {
    return getBooleanValue(key, null);
  }

  /**
   * Returns the Boolean value of a configuration parameter or the default value if no configured value is found.
   *
   * @param key the key of a configuration parameter.
   * @return the Boolean value associated to the {@param key} or the default value if the value is not found.
   */
  default Boolean getBooleanValue(String key, Boolean defaultValue) {
    String value = getStringValue(key);
    if (value != null) {
      return parseBoolean(value);
    }
    return defaultValue;
  }

  /**
   * Returns the Integer value of a configuration parameter.
   *
   * @param key the key of a configuration parameter.
   * @return the Integer value associated to the {@param key} or null if not found.
   */
  default Integer getIntValue(String key) {
    return getIntValue(key, null);
  }

  /**
   * Returns the Integer value of a configuration parameter or the default value if no configured value is found.
   *
   * @param key the key of a configuration parameter.
   * @return the Integer value associated to the {@param key} or the default value if the value is not found.
   */
  default Integer getIntValue(String key, Integer defaultValue) {
    String value = getStringValue(key);
    if (value != null) {
      return parseInt(value);
    }
    return defaultValue;
  }

  /**
   * Returns the Long value of a configuration parameter.
   *
   * @param key the key of a configuration parameter.
   * @return the Long value associated to the {@param key} or null if not found.
   */
  default Long getLongValue(String key) {
    return getLongValue(key, null);
  }

  /**
   * Returns the Long value of a configuration parameter or the default value if no configured value is found.
   *
   * @param key the key of a configuration parameter.
   * @return the Long value associated to the {@param key} or the default value if the value is not found.
   */
  default Long getLongValue(String key, Long defaultValue) {
    String value = getStringValue(key);
    if (value != null) {
      return parseLong(value);
    }
    return defaultValue;
  }

  /**
   * Returns the Double value of a configuration parameter.
   *
   * @param key the key of a configuration parameter.
   * @return the Double value associated to the {@param key} or null if not found.
   */
  default Double getDoubleValue(String key) {
    return getDoubleValue(key, null);
  }

  /**
   * Returns the Double value of a configuration parameter or the default value if no configured value is found.
   *
   * @param key the key of a configuration parameter.
   * @return the Double value associated to the {@param key} or the default value if the value is not found.
   */
  default Double getDoubleValue(String key, Double defaultValue) {
    String value = getStringValue(key);
    if (value != null) {
      return parseDouble(value);
    }
    return defaultValue;
  }

  /**
   * Returns the Duration value of a configuration parameter.
   *
   * @param key the key of a configuration parameter.
   * @return the Duration value associated to the {@param key} or null if not found.
   */
  default Duration getSecondsDurationValue(String key) {
    return getSecondsDurationValue(key, null);
  }

  /**
   * Returns the Duration value of a configuration parameter or the default value if no configured value is found.
   *
   * @param key the key of a configuration parameter.
   * @return the Duration value associated to the {@param key} or the default value if the value is not found.
   */
  default Duration getSecondsDurationValue(String key, Duration defaultValue) {
    Long value = getLongValue(key);
    if (value != null) {
      return ofSeconds(value);
    }
    return defaultValue;
  }

  /**
   * @param doOnConfigurationChanged to execute when a change in the configuration is detected.
   */
  default void doOnConfigurationChanged(Runnable doOnConfigurationChanged) {}

}
