/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.logging.otel.configuration.impl;

import static org.mule.runtime.api.util.IOUtils.getResourceAsUrl;
import static org.mule.runtime.api.util.MuleSystemProperties.ENABLE_OBSERVABILITY_CONFIGURATION_AT_APPLICATION_LEVEL_PROPERTY;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_CONFIGURATION_FILE_NAME;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_CONFIGURATION_FILE_PATH;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_CA_FILE_LOCATION;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_KEY_FILE_LOCATION;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_SERVICE_NAME;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_SERVICE_NAMESPACE;
import static org.mule.runtime.module.observability.configuration.ObservabilityConfigurationFileWatcher.MULE_OBSERVABILITY_CONFIGURATION_WATCHER_DEFAULT_DELAY_PROPERTY;

import static java.io.File.createTempFile;
import static java.lang.Boolean.TRUE;
import static java.lang.System.getProperty;
import static java.lang.System.setProperty;
import static java.nio.file.Files.copy;
import static java.nio.file.Files.createTempDirectory;
import static java.nio.file.Path.of;
import static java.nio.file.Paths.get;
import static java.nio.file.StandardCopyOption.REPLACE_EXISTING;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.equalTo;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.notNullValue;
import static org.hamcrest.Matchers.nullValue;
import static org.junit.jupiter.api.Assertions.assertThrows;

import org.mule.runtime.config.internal.model.dsl.config.PropertyNotFoundException;
import org.mule.runtime.logging.otel.impl.configuration.FileOpenTelemetryLoggingConfiguration;
import org.mule.runtime.module.artifact.api.descriptor.DeployableArtifactDescriptor;
import org.mule.runtime.utils.parameters.SystemProperty;
import org.mule.runtime.utils.probe.PollingProber;
import org.mule.runtime.utils.probe.Probe;

import java.io.File;
import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.Path;
import java.util.Optional;
import java.util.Properties;

import io.qameta.allure.Issue;
import org.junit.jupiter.api.Disabled;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.RegisterExtension;
import org.junit.jupiter.params.ParameterizedClass;
import org.junit.jupiter.params.provider.CsvSource;

@ParameterizedClass(name = "[{index}] {arguments}")
@CsvSource(useHeadersInDisplayName = true, value = {
    "enableConfigInApplication, valuePropertyNonSystemPropertyConfDirectory, valuePropertySystemProperty",
    "true, valueNonSystemProperty, valueSystemProperty",
    "false, valueNonSystemPropertyConfDirectory, valueSystemPropertyConfDirectory"
})
class FileOpenTelemetryLoggingConfigurationTestCase {

  public static final String TEST_CONF_FILE_NAME = "test.conf";
  public static final String EXTERNAL_TEST_CONF_FILE_PATH = "conf/test.conf";
  public static final String SYSTEM_PROPERTY_VALUE = "system_property_value";
  public static final String DEPLOYMENT_PROPERTY_VALUE = "deployment_property_value";
  public static final String TEST_ORIGINAL_CONF_FILE = "test-original.conf";
  public static final String KEY_PROPERTY_NON_SYSTEM_PROPERTY = "keyNonSystemProperty";
  public static final String KEY_PROPERTY_SYSTEM_PROPERTY = "keySystemProperty";
  public static final String KEY_BOOLEAN_VALUE = "booleanValue";
  public static final String KEY_STRING_VALUE_ONE = "stringValueOne";
  public static final String KEY_STRING_VALUE_TWO = "stringValueTwo";
  public static final String NO_KEY_IN_FILE = "noKeyInFile";
  public static final String NON_ARTIFACT_LEVEL_PROPERTY = "nonArtifactLevelProperty";
  public static final String OVER_WRITTEN_ARTIFACT_LEVEL_PROPERTY = "overWrittenArtifactLevelProperty";
  public static final String TEST_OVERWRITTEN_CONF_FILE = "test-overwritten.conf";
  private static final long TIMEOUT_MILLIS = 10000L;

  private final String valueNonSystemProperty;
  private final String valueSystemProperty;

  @RegisterExtension
  public SystemProperty loggingConfigurationFileWatcherDelay =
      new SystemProperty(MULE_OBSERVABILITY_CONFIGURATION_WATCHER_DEFAULT_DELAY_PROPERTY, "1000");

  @RegisterExtension
  public final SystemProperty loggingConfigurationFileName =
      new SystemProperty(MULE_OPEN_TELEMETRY_LOGGING_CONFIGURATION_FILE_NAME,
                         TEST_CONF_FILE_NAME);

  @RegisterExtension
  public final SystemProperty loggingConfigurationFilePath;

  @RegisterExtension
  public final SystemProperty enableConfigInFile;

  @RegisterExtension
  public final SystemProperty overWrittenArtifactLevelSystemProperty;

  @RegisterExtension
  public final SystemProperty nonArtifactLevelSystemProperty;

  public FileOpenTelemetryLoggingConfigurationTestCase(String enableConfigInApplication,
                                                       String valuePropertyNonSystemPropertyConfDirectory,
                                                       String valuePropertySystemProperty)
      throws IOException {
    this.loggingConfigurationFilePath = new SystemProperty(MULE_OPEN_TELEMETRY_LOGGING_CONFIGURATION_FILE_PATH,
                                                           generateExternalConfigurationPath().toAbsolutePath().toString());
    this.enableConfigInFile =
        new SystemProperty(ENABLE_OBSERVABILITY_CONFIGURATION_AT_APPLICATION_LEVEL_PROPERTY, enableConfigInApplication);

    this.overWrittenArtifactLevelSystemProperty =
        new SystemProperty(OVER_WRITTEN_ARTIFACT_LEVEL_PROPERTY, SYSTEM_PROPERTY_VALUE);

    this.nonArtifactLevelSystemProperty = new SystemProperty(NON_ARTIFACT_LEVEL_PROPERTY, SYSTEM_PROPERTY_VALUE);

    this.valueNonSystemProperty = valuePropertyNonSystemPropertyConfDirectory;
    this.valueSystemProperty = valuePropertySystemProperty;
  }

  /**
   * This temporary directory simulates a manual configuration of the config file path. It contains the configuration file to be
   * read in the case of a manual directory configuration.
   *
   * @return Manually configured configuration file path.
   * @throws IOException When the temporary path/file cannot be created.
   */
  private Path generateExternalConfigurationPath() throws IOException {
    Path externalLoggingConfigurationFilePath = createTempDirectory(this.getClass().getName());
    externalLoggingConfigurationFilePath.toFile().deleteOnExit();
    // This temporary file simulates a manual configuration of the config file path and name.
    Path externalConfigFilePath = externalLoggingConfigurationFilePath.resolve(TEST_CONF_FILE_NAME);
    copy(getClass().getClassLoader().getResourceAsStream(EXTERNAL_TEST_CONF_FILE_PATH), externalConfigFilePath, REPLACE_EXISTING);
    externalConfigFilePath.toFile().deleteOnExit();
    return externalLoggingConfigurationFilePath;
  }

  @Test
  void returnsTheValueForANonSystemProperty() {
    FileOpenTelemetryLoggingConfiguration fileOpenTelemetryLoggingConfiguration = getFileOpenTelemetryLoggingConfiguration();
    assertThat(fileOpenTelemetryLoggingConfiguration.getStringValue(KEY_PROPERTY_NON_SYSTEM_PROPERTY), equalTo(
                                                                                                               valueNonSystemProperty));
  }

  @Test
  void returnsTheResolvedSystemProperty() {
    setProperty(valueSystemProperty, SYSTEM_PROPERTY_VALUE);
    FileOpenTelemetryLoggingConfiguration fileOpenTelemetryLoggingConfiguration = getFileOpenTelemetryLoggingConfiguration();
    assertThat(fileOpenTelemetryLoggingConfiguration.getStringValue(KEY_PROPERTY_SYSTEM_PROPERTY),
               equalTo(SYSTEM_PROPERTY_VALUE));
  }

  @Test
  void whenASystemPropertyCannotBeResolvedAnExceptionIsRaised() {
    FileOpenTelemetryLoggingConfiguration fileOpenTelemetryLoggingConfiguration = getFileOpenTelemetryLoggingConfiguration();
    assertThrows(PropertyNotFoundException.class,
                 () -> fileOpenTelemetryLoggingConfiguration.getStringValue(KEY_PROPERTY_SYSTEM_PROPERTY));
  }

  @Test
  void whenNoPropertyIsInTheFileNullValueIsReturned() {
    FileOpenTelemetryLoggingConfiguration fileOpenTelemetryLoggingConfiguration = getFileOpenTelemetryLoggingConfiguration();
    assertThat(fileOpenTelemetryLoggingConfiguration.getStringValue(NO_KEY_IN_FILE), is(nullValue()));
  }

  @Test
  void whenFileIsNotFoundNoPropertyIsFound() {
    FileOpenTelemetryLoggingConfiguration fileNotFoundOpenTelemetryLoggingConfiguration =
        getFileNotFoundOpenTelemetryLoggingConfiguration();
    assertThat(fileNotFoundOpenTelemetryLoggingConfiguration.getStringValue(KEY_PROPERTY_SYSTEM_PROPERTY), is(nullValue()));
    assertThat(fileNotFoundOpenTelemetryLoggingConfiguration.getStringValue(KEY_PROPERTY_NON_SYSTEM_PROPERTY), is(nullValue()));
  }

  @Test
  void readsStringValue() {
    FileOpenTelemetryLoggingConfiguration fileOpenTelemetryLoggingConfiguration = getFileOpenTelemetryLoggingConfiguration();
    String stringValueOne = fileOpenTelemetryLoggingConfiguration.getStringValue(KEY_STRING_VALUE_ONE);
    String stringValueTwo = fileOpenTelemetryLoggingConfiguration.getStringValue(KEY_STRING_VALUE_TWO);

    assertThat(stringValueOne, is("stringValueOne"));
    assertThat(stringValueTwo, is("stringValueTwo"));
  }


  @Test
  void readsBooleanValue() {
    FileOpenTelemetryLoggingConfiguration fileOpenTelemetryLoggingConfiguration = getFileOpenTelemetryLoggingConfiguration();
    String booleanValue = fileOpenTelemetryLoggingConfiguration.getStringValue(KEY_BOOLEAN_VALUE);

    assertThat(booleanValue, is("true"));
  }

  @Test
  void whenValueCorrespondingToPathGetAbsoluteValue() {
    FileOpenTelemetryLoggingConfiguration fileOpenTelemetryLoggingConfiguration = getFileOpenTelemetryLoggingConfiguration();
    Path caFileLocationPath =
        fileOpenTelemetryLoggingConfiguration.getPathValue(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_CA_FILE_LOCATION);
    Path keyFileLocationPath =
        fileOpenTelemetryLoggingConfiguration.getPathValue(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_KEY_FILE_LOCATION);

    assertThat(caFileLocationPath, is(notNullValue()));
    assertThat(keyFileLocationPath, is(notNullValue()));

    assertThat(caFileLocationPath.isAbsolute(), is(TRUE));
    assertThat(keyFileLocationPath.isAbsolute(), is(TRUE));
  }

  @Test
  @Issue("W-20139161")
  void artifactLevelPropertiesMustBeReadFromDeploymentProperties() {
    FileOpenTelemetryLoggingConfiguration fileOpenTelemetryLoggingConfiguration = getFileOpenTelemetryLoggingConfiguration();

    assertThat(fileOpenTelemetryLoggingConfiguration.getStringValue(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_SERVICE_NAMESPACE),
               is("deploymentServiceNamespace"));
    assertThat(fileOpenTelemetryLoggingConfiguration.getStringValue(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_SERVICE_NAME),
               is("deploymentServiceName"));
  }

  @Test
  @Issue("W-20139161")
  void nonArtifactLevelPropertiesMustNotBeReadFromDeploymentProperties() {
    FileOpenTelemetryLoggingConfiguration fileOpenTelemetryLoggingConfiguration = getFileOpenTelemetryLoggingConfiguration();
    assertThat(fileOpenTelemetryLoggingConfiguration.getStringValue(NON_ARTIFACT_LEVEL_PROPERTY), is(SYSTEM_PROPERTY_VALUE));
  }

  @Test
  @Issue("W-20139161")
  void overWrittenArtifactLevelPropertiesMustBeReadFromDeploymentProperties() {
    FileOpenTelemetryLoggingConfiguration fileOpenTelemetryLoggingConfiguration = getFileOpenTelemetryLoggingConfiguration();
    assertThat(getProperty(OVER_WRITTEN_ARTIFACT_LEVEL_PROPERTY), is(SYSTEM_PROPERTY_VALUE));
    assertThat(fileOpenTelemetryLoggingConfiguration.getStringValue(OVER_WRITTEN_ARTIFACT_LEVEL_PROPERTY),
               is(DEPLOYMENT_PROPERTY_VALUE));
  }

  @Test
  @Disabled("To be fixed in W-16676258")
  void configurationFileChanged() throws Exception {
    File file = createTempFile("logging", "test");
    Path testFile = get(file.getPath());
    URI originalConfigFileUri = getResourceAsUrl(TEST_ORIGINAL_CONF_FILE, getClass()).toURI();
    URI overwrittenConfigFileUri = getResourceAsUrl(TEST_OVERWRITTEN_CONF_FILE, getClass()).toURI();
    copy(get(originalConfigFileUri), testFile, REPLACE_EXISTING);

    FileOpenTelemetryLoggingConfiguration fileOpenTelemetryLoggingConfiguration =
        getFileOpenTelemetryLoggingConfiguration(file);

    assertThat(fileOpenTelemetryLoggingConfiguration.getStringValue("key"), equalTo("value"));

    copy(get(overwrittenConfigFileUri), testFile, REPLACE_EXISTING);

    new PollingProber(TIMEOUT_MILLIS, PollingProber.DEFAULT_POLLING_INTERVAL).check(new Probe() {

      @Override
      public boolean isSatisfied() {
        return fileOpenTelemetryLoggingConfiguration.getStringValue("key").equals("value-overwritten");
      }

      @Override
      public String describeFailure() {
        return "";
      }
    });
  }

  private static FileOpenTelemetryLoggingConfiguration getFileOpenTelemetryLoggingConfiguration() {
    // Deployment properties that can be read by the logging configuration.
    Properties deploymentPropertiesStub = new Properties();
    deploymentPropertiesStub.put("mule.openTelemetry.exporter.resource.service.namespace", "deploymentServiceNamespace");
    deploymentPropertiesStub.put("mule.openTelemetry.exporter.resource.service.name", "deploymentServiceName");
    deploymentPropertiesStub.put(NON_ARTIFACT_LEVEL_PROPERTY, DEPLOYMENT_PROPERTY_VALUE);
    deploymentPropertiesStub.put(OVER_WRITTEN_ARTIFACT_LEVEL_PROPERTY, DEPLOYMENT_PROPERTY_VALUE);

    DeployableArtifactDescriptor deployableArtifactDescriptor =
        new DeployableArtifactDescriptor("test-artifact", Optional.of(deploymentPropertiesStub));
    return new FileOpenTelemetryLoggingConfiguration(path -> getConfigurationFile(getResourceAsUrl(TEST_CONF_FILE_NAME,
                                                                                                   FileOpenTelemetryLoggingConfigurationTestCase.class)),
                                                     deployableArtifactDescriptor) {

      @Override
      protected boolean isArtifactLevelProperty(String configurationKey) {
        return super.isArtifactLevelProperty(configurationKey) || configurationKey.equals(OVER_WRITTEN_ARTIFACT_LEVEL_PROPERTY);
      }
    };
  }

  private static File getConfigurationFile(URL filePath) {
    try {
      return new File(filePath.toURI());
    } catch (URISyntaxException e) {
      throw new RuntimeException("Unexpected error while reading configuration file", e);
    }
  }

  private static FileOpenTelemetryLoggingConfiguration getFileOpenTelemetryLoggingConfiguration(File configurationFile) {
    DeployableArtifactDescriptor deployableArtifactDescriptor =
        new DeployableArtifactDescriptor("test-artifact");

    return new FileOpenTelemetryLoggingConfiguration(path -> configurationFile, deployableArtifactDescriptor) {

      @Override
      protected Path getSignalConfigurationFileDirectoryPath() {
        return configurationFile.toPath();
      }
    };
  }

  private static FileOpenTelemetryLoggingConfiguration getFileNotFoundOpenTelemetryLoggingConfiguration() {
    DeployableArtifactDescriptor deployableArtifactDescriptor = new DeployableArtifactDescriptor("test-artifact");
    return new FileOpenTelemetryLoggingConfiguration(path -> null, deployableArtifactDescriptor) {

      @Override
      protected Path getSignalConfigurationFileDirectoryPath() {
        return of("unexistentPath");
      }

    };
  }

}
