/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.logging.otel.impl.export.log4j;

import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_ENABLED;
import static org.mule.runtime.logging.otel.impl.OpenTelemetryLoggingSDKFactory.getOpenTelemetryLoggingSDK;

import static java.lang.Boolean.getBoolean;

import static org.apache.logging.log4j.Level.ALL;

import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.logging.otel.impl.configuration.LogRecordExporterConfiguratorException;
import org.mule.runtime.module.artifact.api.classloader.DirectoryResourceLocator;
import org.mule.runtime.module.artifact.api.descriptor.ClassLoaderConfiguration;
import org.mule.runtime.module.artifact.api.descriptor.DeployableArtifactDescriptor;
import org.mule.runtime.module.observability.configuration.ObservabilitySignalConfigurationFileFinder;

import java.io.File;
import java.net.URISyntaxException;
import java.net.URL;

import io.opentelemetry.instrumentation.log4j.appender.v2_17.OpenTelemetryAppender;
import org.apache.logging.log4j.core.Appender;
import org.apache.logging.log4j.core.LoggerContext;

/**
 * Allows configuration of a Log4J -> OpenTelemetry logging bridge.
 *
 * @since 4.10.0
 */
public class OpenTelemetryLog4JBridge {

  /**
   * An Appender added through #addOpenTelemetryLogging will have this suffix as part of its name.
   *
   * @see Appender#getName()
   */
  public static final String OPEN_TELEMETRY_APPENDER_NAME_SUFFIX = " OpenTelemetry logs";

  public static final String MULE_CONTAINER_APPENDER_NAME = "mule-container";

  private OpenTelemetryLog4JBridge() {}

  public static void addOpenTelemetryLogging(LoggerContext deployableArtifactLoggerContext,
                                             DeployableArtifactDescriptor deployableArtifactDescriptor)
      throws LogRecordExporterConfiguratorException {
    if (getBoolean(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_ENABLED)) {
      try {
        doAddOpenTelemetryLogging(deployableArtifactLoggerContext, deployableArtifactDescriptor,
                                  new ClassloaderConfigurationObservabilitySignalConfigurationFileFinder(deployableArtifactDescriptor));
      } catch (LogRecordExporterConfiguratorException e) {
        throw new MuleRuntimeException(createStaticMessage("Error configuring OpenTelemetry logging bridge"), e);
      }
    }
  }

  /**
   * Adds an OpenTelemetry LOG4J appender to a {@link LoggerContext} that belongs to the mule container. The appender is a bridge
   * between log4j and Open Telemetry SDK. The SDK will be already configured to export OpenTelemetry logs and will start the
   * export immediately. The lifecycle of the Open Telemetry SDK will be tied to the LoggerContext lifecycle, supporting
   * reconfiguration, start, stop and shutdown methods.
   *
   * @param muleContainerLoggerContext              The {@link LoggerContext} The mule container logger context.
   * @param muleContainerLoggingConfigurationFinder Finder that can access the mule container logging signal configuration.
   * @throws MuleRuntimeException If the open telemetry logging cannot be configured.
   */
  public static void addOpenTelemetryLogging(LoggerContext muleContainerLoggerContext,
                                             ObservabilitySignalConfigurationFileFinder muleContainerLoggingConfigurationFinder) {
    if (getBoolean(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_ENABLED)) {
      try {
        doAddOpenTelemetryLogging(muleContainerLoggerContext, null, muleContainerLoggingConfigurationFinder);
      } catch (LogRecordExporterConfiguratorException e) {
        throw new MuleRuntimeException(createStaticMessage("Error configuring OpenTelemetry logging bridge"), e);
      }
    }
  }

  private static void doAddOpenTelemetryLogging(LoggerContext muleLoggerContext, DeployableArtifactDescriptor artifactDescriptor,
                                                ObservabilitySignalConfigurationFileFinder loggingConfigurationFinder)
      throws LogRecordExporterConfiguratorException {
    Appender openTelemetryAppender =
        getOpenTelemetryLog4jAppender(muleLoggerContext, artifactDescriptor, loggingConfigurationFinder);
    muleLoggerContext.getConfiguration().addAppender(openTelemetryAppender);
    openTelemetryAppender.start();
    // Take advantage of the log4j additivity to ensure that all loggers will use the appender.
    muleLoggerContext.getConfiguration().getRootLogger().addAppender(openTelemetryAppender, ALL, null);
    // Non - additive loggers are also taken into account
    muleLoggerContext.getConfiguration().getLoggers().forEach((logger, loggerConfig) -> {
      if (!loggerConfig.isAdditive()) {
        loggerConfig.addAppender(openTelemetryAppender, ALL, null);
      }
    });
  }

  /**
   * Adds a Log4J OpenTelemetry appender to a logger context.
   *
   * @param loggerContext      The LoggerContext to add the appender to.
   * @param artifactDescriptor The descriptor of the artifact that owns the logger context. Can be null, meaning that the logger
   *                           context belongs to the mule container. \ * @return The Log4J OpenTelemetry appender.
   */
  private static Appender getOpenTelemetryLog4jAppender(LoggerContext loggerContext,
                                                        DeployableArtifactDescriptor artifactDescriptor,
                                                        ObservabilitySignalConfigurationFileFinder loggingConfigurationFinder) {
    OpenTelemetryAppender openTelemetryAppender = OpenTelemetryAppender.builder()
        .setConfiguration(loggerContext.getConfiguration())
        .setName(resolveAppenderName(artifactDescriptor))
        // Full MDC is captured by default (configuration is not yet necessary).
        .setCaptureContextDataAttributes("*")
        .setCaptureExperimentalAttributes(true)
        .build();
    return DedicatedSdkOpenTelemetryAppender.builder(openTelemetryAppender)
        .withOpenTelemetryLoggingSdkProvider(() -> getOpenTelemetryLoggingSDK(artifactDescriptor,
                                                                              loggingConfigurationFinder))
        .build();
  }

  private static String resolveAppenderName(DeployableArtifactDescriptor artifactDescriptor) {
    return (artifactDescriptor != null ? artifactDescriptor.getName() : MULE_CONTAINER_APPENDER_NAME)
        .concat(OPEN_TELEMETRY_APPENDER_NAME_SUFFIX);
  }

  /**
   * {@link ObservabilitySignalConfigurationFileFinder} that searches for artifact configuration files using it artifact's
   * classloader configuration. Useful to avoid classloader referencing and the potential leaks that it might imply.
   */
  private static class ClassloaderConfigurationObservabilitySignalConfigurationFileFinder
      implements ObservabilitySignalConfigurationFileFinder {

    private final DirectoryResourceLocator artifactResourcesDirectoryLocator;

    public ClassloaderConfigurationObservabilitySignalConfigurationFileFinder(DeployableArtifactDescriptor deployableArtifactDescriptor) {
      ClassLoaderConfiguration classLoaderConfiguration = deployableArtifactDescriptor.getClassLoaderConfiguration();
      // The first entry of the classloader configuration URLs is the artifact root folder.
      if (classLoaderConfiguration != null && classLoaderConfiguration.getUrls() != null
          && classLoaderConfiguration.getUrls().length > 0) {
        this.artifactResourcesDirectoryLocator = new DirectoryResourceLocator(classLoaderConfiguration.getUrls()[0].getPath());
      } else {
        this.artifactResourcesDirectoryLocator = new DirectoryResourceLocator();
      }
    }

    @Override
    public File getResource(String path) {
      try {
        URL resource = artifactResourcesDirectoryLocator.findLocalResource(path);
        return resource != null ? new File(resource.toURI()) : null;
      } catch (URISyntaxException e) {
        return null;
      }
    }

  }

}
