/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.logging.otel.impl.configuration;

import static org.mule.runtime.api.i18n.I18nMessageFactory.createStaticMessage;
import static org.mule.runtime.container.api.MuleFoldersUtil.getConfFolder;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_CONFIGURATION_FILE_NAME;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_CONFIGURATION_FILE_PATH;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_DEFAULT_CONFIGURATION_FILE_NAME;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_SERVICE_NAME;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_SERVICE_NAMESPACE;

import static java.lang.String.format;
import static java.lang.System.getProperty;
import static java.nio.file.Paths.get;
import static java.util.List.of;

import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.api.exception.MuleRuntimeException;
import org.mule.runtime.module.artifact.api.descriptor.DeployableArtifactDescriptor;
import org.mule.runtime.module.observability.AbstractFileObservabilitySignalConfiguration;
import org.mule.runtime.module.observability.configuration.ObservabilitySignalConfigurationFileFinder;

import java.io.File;
import java.nio.file.Path;
import java.util.List;

import org.slf4j.Logger;

public class FileOpenTelemetryLoggingConfiguration extends AbstractFileObservabilitySignalConfiguration {

  private static final Logger logger = getLogger(FileOpenTelemetryLoggingConfiguration.class);
  private static final List<String> artifactLevelConfigurationProperties =
      of(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_SERVICE_NAME, MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_SERVICE_NAMESPACE);

  public FileOpenTelemetryLoggingConfiguration(ObservabilitySignalConfigurationFileFinder artifactResourceFinder,
                                               DeployableArtifactDescriptor deployableArtifactDescriptor) {
    super(artifactResourceFinder, deployableArtifactDescriptor);
  }


  @Override
  protected void onConfigurationFileNotFound() {
    logger.atInfo().log("Logging exporter configuration file named '{}' not found {}. Using default configuration.",
                        getSignalConfigurationFileName(), getConfigurationFileLocations());
  }

  @Override
  protected void onConfigurationFileLoadError(Exception error, File configurationFile) {
    // Logging is considered critical feature, therefore we rise the configuration parse error.
    throw new MuleRuntimeException(createStaticMessage(format("Parsing error in the logging exporter configuration file '%s'.",
                                                              configurationFile.getAbsolutePath()),
                                                       error));
  }

  @Override
  protected Path getSignalConfigurationFileDirectoryPath() {
    return get(getProperty(MULE_OPEN_TELEMETRY_LOGGING_CONFIGURATION_FILE_PATH,
                           getConfFolder().getAbsolutePath())).toAbsolutePath();
  }

  @Override
  protected boolean isArtifactLevelProperty(String configurationKey) {
    return artifactLevelConfigurationProperties.contains(configurationKey);
  }

  @Override
  protected String getSignalConfigurationFileName() {
    return getProperty(MULE_OPEN_TELEMETRY_LOGGING_CONFIGURATION_FILE_NAME,
                       MULE_OPEN_TELEMETRY_LOGGING_DEFAULT_CONFIGURATION_FILE_NAME);
  }

}
