/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.logging.otel.impl.export.batch;

import org.mule.runtime.logging.otel.impl.configuration.OpenTelemetryLoggingExporterBackpressureStrategy;

import static java.lang.Thread.currentThread;

import static org.slf4j.LoggerFactory.getLogger;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;

import io.opentelemetry.api.common.AttributeKey;
import io.opentelemetry.api.common.Attributes;
import io.opentelemetry.api.metrics.LongCounter;
import io.opentelemetry.api.metrics.Meter;
import io.opentelemetry.api.metrics.MeterProvider;
import io.opentelemetry.context.Context;
import io.opentelemetry.sdk.common.CompletableResultCode;
import io.opentelemetry.sdk.internal.DaemonThreadFactory;
import io.opentelemetry.sdk.logs.LogRecordProcessor;
import io.opentelemetry.sdk.logs.ReadWriteLogRecord;
import io.opentelemetry.sdk.logs.data.LogRecordData;
import io.opentelemetry.sdk.logs.export.BatchLogRecordProcessor;
import io.opentelemetry.sdk.logs.export.LogRecordExporter;
import org.slf4j.Logger;

/**
 * This class is an almost exact copy of Open Telemetry SDK's {@link BatchLogRecordProcessor} but it has a twist: In the case of
 * the batch queue being full, it can either block until space is freed or drop the log record. It will track such block/drop
 * events and report them through a metrics signal.
 */
// TODO: W-19294623: Generalize BlockingBatchLogRecordProcessor.
public final class BlockingBatchLogRecordProcessor implements LogRecordProcessor {

  private static final String WORKER_THREAD_NAME =
      BlockingBatchLogRecordProcessor.class.getSimpleName() + "_WorkerThread";
  private static final AttributeKey<String> LOG_RECORD_PROCESSOR_TYPE_LABEL =
      AttributeKey.stringKey("processorType");
  private static final AttributeKey<Boolean> LOG_RECORD_PROCESSOR_DROPPED_LABEL =
      AttributeKey.booleanKey("dropped");
  private static final String LOG_RECORD_PROCESSOR_TYPE_VALUE =
      BlockingBatchLogRecordProcessor.class.getSimpleName();

  private final Worker worker;
  private final AtomicBoolean isShutdown = new AtomicBoolean(false);

  /**
   * Returns a new Builder for {@link BlockingBatchLogRecordProcessor}.
   *
   * @param logRecordExporter the {@link LogRecordExporter} to which the Logs are pushed
   * @return a new {@link BlockingBatchLogRecordProcessor}.
   * @throws NullPointerException if the {@code logRecordExporter} is {@code null}.
   */
  public static BlockingBatchLogRecordProcessorBuilder builder(LogRecordExporter logRecordExporter) {
    return new BlockingBatchLogRecordProcessorBuilder(logRecordExporter);
  }

  BlockingBatchLogRecordProcessor(
                                  LogRecordExporter logRecordExporter,
                                  MeterProvider meterProvider,
                                  long scheduleDelayNanos,
                                  int maxQueueSize,
                                  int maxExportBatchSize,
                                  long exporterTimeoutNanos,
                                  OpenTelemetryLoggingExporterBackpressureStrategy backpressureStrategy) {
    this.worker =
        new Worker(
                   logRecordExporter,
                   meterProvider,
                   scheduleDelayNanos,
                   maxExportBatchSize,
                   exporterTimeoutNanos,
                   new ArrayBlockingQueue<>(maxQueueSize),
                   backpressureStrategy);
    // TODO: W-19294666: Use Scheduler Service threads for signal exporters.
    Thread workerThread = new DaemonThreadFactory(WORKER_THREAD_NAME).newThread(worker);
    workerThread.start();
  }

  @Override
  public void onEmit(Context context, ReadWriteLogRecord logRecord) {
    if (logRecord == null) {
      return;
    }
    worker.addLog(logRecord);
  }

  @Override
  public CompletableResultCode shutdown() {
    if (isShutdown.getAndSet(true)) {
      return CompletableResultCode.ofSuccess();
    }
    return worker.shutdown();
  }

  @Override
  public CompletableResultCode forceFlush() {
    return worker.forceFlush();
  }

  /**
   * Return the processor's configured {@link LogRecordExporter}.
   *
   * @since 1.37.0
   */
  public LogRecordExporter getLogRecordExporter() {
    return worker.logRecordExporter;
  }

  // Visible for testing
  List<LogRecordData> getBatch() {
    return worker.batch;
  }

  @Override
  public String toString() {
    return "BatchLogRecordProcessor{"
        + "logRecordExporter="
        + worker.logRecordExporter
        + ", scheduleDelayNanos="
        + worker.scheduleDelayNanos
        + ", maxExportBatchSize="
        + worker.maxExportBatchSize
        + ", exporterTimeoutNanos="
        + worker.exporterTimeoutNanos
        + '}';
  }

  // Worker is a thread that batches multiple logs and calls the registered LogRecordExporter to
  // export
  // the data.
  private static final class Worker implements Runnable {

    private static final Logger logger = getLogger(Worker.class);

    private final LongCounter processedLogsCounter;
    private final LongCounter delayedLogs;
    private final Attributes droppedAttrs;
    private final Attributes exportedAttrs;

    private final LogRecordExporter logRecordExporter;
    private final long scheduleDelayNanos;
    private final int maxExportBatchSize;
    private final long exporterTimeoutNanos;
    private final boolean isBlockWhenFullQueue;

    private long nextExportTime;

    private final BlockingQueue<ReadWriteLogRecord> queue;
    // When waiting on the logs queue, exporter thread sets this atomic to the number of more
    // logs it needs before doing an export. Writer threads would then wait for the queue to reach
    // logsNeeded size before notifying the exporter thread about new entries.
    // Integer.MAX_VALUE is used to imply that exporter thread is not expecting any signal. Since
    // exporter thread doesn't expect any signal initially, this value is initialized to
    // Integer.MAX_VALUE.
    private final AtomicInteger logsNeeded = new AtomicInteger(Integer.MAX_VALUE);
    private final BlockingQueue<Boolean> signal;
    private final AtomicReference<CompletableResultCode> flushRequested = new AtomicReference<>();
    private volatile boolean continueWork = true;
    private final ArrayList<LogRecordData> batch;

    private Worker(
                   LogRecordExporter logRecordExporter,
                   MeterProvider meterProvider,
                   long scheduleDelayNanos,
                   int maxExportBatchSize,
                   long exporterTimeoutNanos,
                   BlockingQueue<ReadWriteLogRecord> queue,
                   OpenTelemetryLoggingExporterBackpressureStrategy backpressureStrategy) {
      this.logRecordExporter = logRecordExporter;
      this.scheduleDelayNanos = scheduleDelayNanos;
      this.maxExportBatchSize = maxExportBatchSize;
      this.exporterTimeoutNanos = exporterTimeoutNanos;
      this.queue = queue;
      this.isBlockWhenFullQueue = backpressureStrategy == OpenTelemetryLoggingExporterBackpressureStrategy.BLOCK;
      this.signal = new ArrayBlockingQueue<>(1);
      Meter meter = meterProvider.meterBuilder("io.opentelemetry.sdk.logs").build();
      meter
          .gaugeBuilder("queueSize")
          .ofLongs()
          .setDescription("The number of items queued")
          .setUnit("1")
          .buildWithCallback(
                             result -> result.record(
                                                     queue.size(),
                                                     Attributes.of(
                                                                   LOG_RECORD_PROCESSOR_TYPE_LABEL,
                                                                   LOG_RECORD_PROCESSOR_TYPE_VALUE)));
      processedLogsCounter =
          meter
              .counterBuilder("processedLogs")
              .setUnit("1")
              .setDescription(
                              "The number of logs processed by the BatchLogRecordProcessor. "
                                  + "[dropped=true if they were dropped due to high throughput]")
              .build();
      delayedLogs =
          meter
              .counterBuilder("delayedLogs")
              .setUnit("1")
              .setDescription(
                              "The number of logs that spent time waiting to enter the BatchLogRecordProcessor because of the export queue being full. ")
              .build();
      droppedAttrs =
          Attributes.of(
                        LOG_RECORD_PROCESSOR_TYPE_LABEL,
                        LOG_RECORD_PROCESSOR_TYPE_VALUE,
                        LOG_RECORD_PROCESSOR_DROPPED_LABEL,
                        true);
      exportedAttrs =
          Attributes.of(
                        LOG_RECORD_PROCESSOR_TYPE_LABEL,
                        LOG_RECORD_PROCESSOR_TYPE_VALUE,
                        LOG_RECORD_PROCESSOR_DROPPED_LABEL,
                        false);

      this.batch = new ArrayList<>(this.maxExportBatchSize);
    }

    private void addLog(ReadWriteLogRecord logData) {
      // If space is available, we do not block
      if (!queue.offer(logData)) {
        if (isBlockWhenFullQueue) {
          try {
            // No space available: Blocking put + track the block
            delayedLogs.add(1);
            queue.put(logData);
          } catch (InterruptedException e) {
            // Unexpected error: discard the log record + track the drop
            processedLogsCounter.add(1, droppedAttrs);
            currentThread().interrupt();
          }
        } else {
          // No space available: discard the log record + track the drop
          processedLogsCounter.add(1, droppedAttrs);
        }
      } else {
        if (queue.size() >= logsNeeded.get()) {
          signal.offer(true);
        }
      }
    }

    @Override
    public void run() {
      updateNextExportTime();

      while (continueWork) {
        if (flushRequested.get() != null) {
          flush();
        }
        while (!queue.isEmpty() && batch.size() < maxExportBatchSize) {
          batch.add(queue.poll().toLogRecordData());
        }
        if (batch.size() >= maxExportBatchSize || System.nanoTime() >= nextExportTime) {
          exportCurrentBatch();
          updateNextExportTime();
        }
        if (queue.isEmpty()) {
          try {
            long pollWaitTime = nextExportTime - System.nanoTime();
            if (pollWaitTime > 0) {
              logsNeeded.set(maxExportBatchSize - batch.size());
              signal.poll(pollWaitTime, TimeUnit.NANOSECONDS);
              logsNeeded.set(Integer.MAX_VALUE);
            }
          } catch (InterruptedException e) {
            currentThread().interrupt();
            return;
          }
        }
      }
    }

    private void flush() {
      int logsToFlush = queue.size();
      while (logsToFlush > 0) {
        ReadWriteLogRecord logRecord = queue.poll();
        assert logRecord != null;
        batch.add(logRecord.toLogRecordData());
        logsToFlush--;
        if (batch.size() >= maxExportBatchSize) {
          exportCurrentBatch();
        }
      }
      exportCurrentBatch();
      CompletableResultCode flushResult = flushRequested.get();
      if (flushResult != null) {
        flushResult.succeed();
        flushRequested.set(null);
      }
    }

    private void updateNextExportTime() {
      nextExportTime = System.nanoTime() + scheduleDelayNanos;
    }

    private CompletableResultCode shutdown() {
      CompletableResultCode result = new CompletableResultCode();

      CompletableResultCode flushResult = forceFlush();
      flushResult.whenComplete(
                               () -> {
                                 continueWork = false;
                                 CompletableResultCode shutdownResult = logRecordExporter.shutdown();
                                 shutdownResult.whenComplete(
                                                             () -> {
                                                               if (!flushResult.isSuccess() || !shutdownResult.isSuccess()) {
                                                                 result.fail();
                                                               } else {
                                                                 result.succeed();
                                                               }
                                                             });
                               });

      return result;
    }

    private CompletableResultCode forceFlush() {
      CompletableResultCode flushResult = new CompletableResultCode();
      // we set the atomic here to trigger the worker loop to do a flush of the entire queue.
      if (flushRequested.compareAndSet(null, flushResult)) {
        signal.offer(true);
      }
      CompletableResultCode possibleResult = flushRequested.get();
      // there's a race here where the flush happening in the worker loop could complete before we
      // get what's in the atomic. In that case, just return success, since we know it succeeded in
      // the interim.
      return possibleResult == null ? CompletableResultCode.ofSuccess() : possibleResult;
    }

    private void exportCurrentBatch() {
      if (batch.isEmpty()) {
        return;
      }

      try {
        CompletableResultCode result =
            logRecordExporter.export(Collections.unmodifiableList(batch));
        result.join(exporterTimeoutNanos, TimeUnit.NANOSECONDS);
        if (result.isSuccess()) {
          processedLogsCounter.add(batch.size(), exportedAttrs);
        } else {
          if (result.getFailureThrowable() != null) {
            logger.warn("Batch logging export failed.", result.getFailureThrowable());
          } else {
            logger.warn("Batch logging export failed without error.");
          }
        }
      } catch (RuntimeException e) {
        logger.warn("Batch logging export threw an Exception", e);
      } finally {
        batch.clear();
      }
    }
  }


}
