/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.logging.otel.configuration.impl.export;

import static java.util.concurrent.Executors.newFixedThreadPool;

import static org.mockito.Mockito.mock;

import org.mule.runtime.logging.otel.impl.export.batch.BlockingBatchLogRecordProcessor;
import org.mule.runtime.logging.otel.impl.export.sniffer.ExportedLogRecordSniffer;
import org.mule.runtime.logging.otel.impl.export.sniffer.SniffedLogRecordExporter;

import java.util.Collection;
import java.util.concurrent.ExecutorService;

import io.opentelemetry.context.Context;
import io.opentelemetry.sdk.common.CompletableResultCode;
import io.opentelemetry.sdk.logs.ReadWriteLogRecord;
import io.opentelemetry.sdk.logs.data.LogRecordData;
import io.opentelemetry.sdk.logs.export.LogRecordExporter;
import io.opentelemetry.sdk.metrics.SdkMeterProvider;
import io.opentelemetry.sdk.metrics.data.LongPointData;
import io.opentelemetry.sdk.metrics.data.MetricData;
import io.opentelemetry.sdk.metrics.data.MetricDataType;
import io.opentelemetry.sdk.metrics.data.PointData;
import io.opentelemetry.sdk.testing.exporter.InMemoryMetricReader;
import org.junit.jupiter.api.Test;
import org.mule.runtime.utils.probe.PollingProber;
import org.mule.runtime.utils.probe.Probe;

class BlockingBatchLogRecordProcessorTestCase {

  private static final long TIMEOUT_MILLIS = 10000L;

  @Test
  void testWhenQueueFullThenLogRecordWaitsForSpace() {
    ExecutorService executor = newFixedThreadPool(1);
    // Create in-memory metric reader
    InMemoryMetricReader inMemoryMetricReader = InMemoryMetricReader.create();
    try (BlockingNoOpLogRecordExporter blockingNoOpLogRecordExporter = new BlockingNoOpLogRecordExporter();
        SdkMeterProvider meterProvider = SdkMeterProvider.builder()
            .registerMetricReader(inMemoryMetricReader)
            .build();
        SniffedLogRecordExporter sniffedLogRecordExporter = new SniffedLogRecordExporter(blockingNoOpLogRecordExporter);
        BlockingBatchLogRecordProcessor blockingBatchLogRecordProcessor =
            BlockingBatchLogRecordProcessor.builder(sniffedLogRecordExporter)
                .setMeterProvider(meterProvider)
                .setMaxQueueSize(1)
                .setMaxExportBatchSize(1)
                .build()) {
      ExportedLogRecordSniffer exportedLogRecordSniffer = sniffedLogRecordExporter.getExportedLogRecordSniffer();
      // This first log record will block the exporter.
      blockingBatchLogRecordProcessor.onEmit(mock(Context.class), mock(ReadWriteLogRecord.class));
      blockingNoOpLogRecordExporter.waitUntilIsBlocked();
      // This second log record will fill the export queue.
      blockingBatchLogRecordProcessor.onEmit(mock(Context.class), mock(ReadWriteLogRecord.class));
      // This third log record will have to wait (therefore we launch it on a different thread.
      executor.submit(() -> blockingBatchLogRecordProcessor.onEmit(mock(Context.class), mock(ReadWriteLogRecord.class)));
      // Assert that the waiting log record is causing the delayedLogs metrics to increase.
      new PollingProber(TIMEOUT_MILLIS, PollingProber.DEFAULT_POLLING_INTERVAL).check(new Probe() {

        @Override
        public boolean isSatisfied() {
          return inMemoryMetricReader.collectAllMetrics().stream()
              .anyMatch(metricData -> metricData.getName().equals("delayedLogs") && getCounterValue(metricData) == 1);
        }

        @Override
        public String describeFailure() {
          return "Delayed log record was not reported at the delayedLogs metric.";
        }
      });
      // Unblock the exporter in order to perform the export of the three log records.
      blockingNoOpLogRecordExporter.unblock();
      // Assert that all the log records were exported (including the one that waited to enter the export queue)
      new PollingProber(TIMEOUT_MILLIS, PollingProber.DEFAULT_POLLING_INTERVAL).check(new Probe() {

        @Override
        public boolean isSatisfied() {
          return exportedLogRecordSniffer.getSniffedLogRecords().size() == 3;
        }

        @Override
        public String describeFailure() {
          return "Not all the log records were exported. Expected: 3, Actual: "
              + exportedLogRecordSniffer.getSniffedLogRecords().size();
        }
      });
    }
  }

  private long getCounterValue(MetricData metricData) {
    long value = 0;
    if (metricData.getType() == MetricDataType.LONG_SUM) {
      for (PointData point : metricData.getData().getPoints()) {
        if (point instanceof LongPointData) {
          value = value + ((LongPointData) point).getValue();
        }
      }
    }
    return value;
  }

  private static final class BlockingNoOpLogRecordExporter implements LogRecordExporter {

    final Object monitor = new Object();

    private enum State {
      WAIT_TO_BLOCK, BLOCKED, UNBLOCKED
    }

    State state = State.WAIT_TO_BLOCK;

    @Override
    public CompletableResultCode export(Collection<LogRecordData> logs) {
      synchronized (monitor) {
        while (state != State.UNBLOCKED) {
          try {
            state = State.BLOCKED;
            // Some threads may wait for Blocked State.
            monitor.notifyAll();
            monitor.wait();
          } catch (InterruptedException e) {
            // Do nothing
          }
        }
      }
      return CompletableResultCode.ofSuccess();
    }

    @Override
    public CompletableResultCode flush() {
      return CompletableResultCode.ofSuccess();
    }

    private void waitUntilIsBlocked() {
      synchronized (monitor) {
        while (state != State.BLOCKED) {
          try {
            monitor.wait();
          } catch (InterruptedException e) {
            // Do nothing
          }
        }
      }
    }

    @Override
    public CompletableResultCode shutdown() {
      // Do nothing;
      return CompletableResultCode.ofSuccess();
    }

    @Override
    public void close() {
      LogRecordExporter.super.close();
    }

    private void unblock() {
      synchronized (monitor) {
        state = State.UNBLOCKED;
        monitor.notifyAll();
      }
    }
  }
}
