/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.logging.otel.impl.export.batch;

import static java.util.Objects.requireNonNull;

import static org.slf4j.LoggerFactory.getLogger;

import org.mule.runtime.logging.otel.impl.configuration.OpenTelemetryLoggingExporterBackpressureStrategy;

import java.time.Duration;
import java.util.concurrent.TimeUnit;

import io.opentelemetry.api.metrics.MeterProvider;
import io.opentelemetry.sdk.logs.export.LogRecordExporter;
import org.slf4j.Logger;

/**
 * Builder class for {@link BlockingBatchLogRecordProcessor}.
 */
public final class BlockingBatchLogRecordProcessorBuilder {

  private static final Logger logger =
      getLogger(BlockingBatchLogRecordProcessorBuilder.class);

  // Visible for testing
  static final long DEFAULT_SCHEDULE_DELAY_MILLIS = 1000;
  // Visible for testing
  static final int DEFAULT_MAX_QUEUE_SIZE = 2048;
  // Visible for testing
  static final int DEFAULT_MAX_EXPORT_BATCH_SIZE = 512;
  // Visible for testing
  static final int DEFAULT_EXPORT_TIMEOUT_MILLIS = 30_000;

  private final LogRecordExporter logRecordExporter;
  private long scheduleDelayNanos = TimeUnit.MILLISECONDS.toNanos(DEFAULT_SCHEDULE_DELAY_MILLIS);
  private int maxQueueSize = DEFAULT_MAX_QUEUE_SIZE;
  private int maxExportBatchSize = DEFAULT_MAX_EXPORT_BATCH_SIZE;
  private long exporterTimeoutNanos = TimeUnit.MILLISECONDS.toNanos(DEFAULT_EXPORT_TIMEOUT_MILLIS);
  private MeterProvider meterProvider = MeterProvider.noop();
  private OpenTelemetryLoggingExporterBackpressureStrategy backpressureStrategy =
      OpenTelemetryLoggingExporterBackpressureStrategy.BLOCK;

  BlockingBatchLogRecordProcessorBuilder(LogRecordExporter logRecordExporter) {
    this.logRecordExporter = requireNonNull(logRecordExporter, "logRecordExporter");
  }

  /**
   * Sets the delay interval between two consecutive exports. If unset, defaults to {@value DEFAULT_SCHEDULE_DELAY_MILLIS}ms.
   */
  public BlockingBatchLogRecordProcessorBuilder setScheduleDelay(long delay, TimeUnit unit) {
    requireNonNull(unit, "unit");
    checkArgument(delay >= 0, "delay must be non-negative");
    scheduleDelayNanos = unit.toNanos(delay);
    return this;
  }

  /**
   * Sets the delay interval between two consecutive exports. If unset, defaults to {@value DEFAULT_SCHEDULE_DELAY_MILLIS}ms.
   */
  public BlockingBatchLogRecordProcessorBuilder setScheduleDelay(Duration delay) {
    requireNonNull(delay, "delay");
    return setScheduleDelay(delay.toNanos(), TimeUnit.NANOSECONDS);
  }

  // Visible for testing
  long getScheduleDelayNanos() {
    return scheduleDelayNanos;
  }

  /**
   * Sets the maximum time an export will be allowed to run before being cancelled. If unset, defaults to
   * {@value DEFAULT_EXPORT_TIMEOUT_MILLIS}ms.
   */
  public BlockingBatchLogRecordProcessorBuilder setExporterTimeout(long timeout, TimeUnit unit) {
    requireNonNull(unit, "unit");
    checkArgument(timeout >= 0, "timeout must be non-negative");
    exporterTimeoutNanos = timeout == 0 ? Long.MAX_VALUE : unit.toNanos(timeout);
    return this;
  }

  /**
   * Sets the maximum time an export will be allowed to run before being cancelled. If unset, defaults to
   * {@value DEFAULT_EXPORT_TIMEOUT_MILLIS}ms.
   */
  public BlockingBatchLogRecordProcessorBuilder setExporterTimeout(Duration timeout) {
    requireNonNull(timeout, "timeout");
    return setExporterTimeout(timeout.toNanos(), TimeUnit.NANOSECONDS);
  }

  // Visible for testing
  long getExporterTimeoutNanos() {
    return exporterTimeoutNanos;
  }

  /**
   * Sets the maximum number of Logs that are kept in the queue before start dropping. More memory than this value may be
   * allocated to optimize queue access.
   *
   * <p>
   * Default value is {@code 2048}.
   *
   * @param maxQueueSize the maximum number of Logs that are kept in the queue before start dropping.
   * @return this.
   * @throws IllegalArgumentException if {@code maxQueueSize} is not positive.
   * @see BlockingBatchLogRecordProcessorBuilder#DEFAULT_MAX_QUEUE_SIZE
   */
  public BlockingBatchLogRecordProcessorBuilder setMaxQueueSize(int maxQueueSize) {
    checkArgument(maxQueueSize > 0, "maxQueueSize must be positive.");
    this.maxQueueSize = maxQueueSize;
    return this;
  }

  /**
   * Sets the maximum batch size for every export. This must be smaller or equal to {@code
   * maxQueueSize}.
   *
   * <p>
   * Default value is {@code 512}.
   *
   * @param maxExportBatchSize the maximum batch size for every export.
   * @return this.
   * @see BlockingBatchLogRecordProcessorBuilder#DEFAULT_MAX_EXPORT_BATCH_SIZE
   */
  public BlockingBatchLogRecordProcessorBuilder setMaxExportBatchSize(int maxExportBatchSize) {
    checkArgument(maxExportBatchSize > 0, "maxExportBatchSize must be positive.");
    this.maxExportBatchSize = maxExportBatchSize;
    return this;
  }

  /**
   * Sets the {@link MeterProvider} to use to collect metrics related to batch export. If not set, metrics will not be collected.
   */
  public BlockingBatchLogRecordProcessorBuilder setMeterProvider(MeterProvider meterProvider) {
    requireNonNull(meterProvider, "meterProvider");
    this.meterProvider = meterProvider;
    return this;
  }

  // Visible for testing
  int getMaxExportBatchSize() {
    return maxExportBatchSize;
  }

  /**
   * Sets the backpressure strategy used to treat logging records that cannot enter the export queue because it is full.
   *
   * @param backpressureStrategy The backpressure strategy.
   * @return The configured BlockingBatchLogRecordProcessorBuilder.
   */
  public BlockingBatchLogRecordProcessorBuilder setBackpressureStrategy(OpenTelemetryLoggingExporterBackpressureStrategy backpressureStrategy) {
    this.backpressureStrategy = backpressureStrategy;
    return this;
  }

  /**
   * Returns a new {@link BlockingBatchLogRecordProcessor} that batches, then forwards them to the given
   * {@code logRecordExporter}.
   *
   * @return a new {@link BlockingBatchLogRecordProcessor}.
   */
  public BlockingBatchLogRecordProcessor build() {
    if (maxExportBatchSize > maxQueueSize) {
      logger.warn(
                  "maxExportBatchSize should not exceed maxQueueSize. Setting maxExportBatchSize to {} instead of {}",
                  maxQueueSize, maxExportBatchSize);
      maxExportBatchSize = maxQueueSize;
    }
    return new BlockingBatchLogRecordProcessor(
                                               logRecordExporter,
                                               meterProvider,
                                               scheduleDelayNanos,
                                               maxQueueSize,
                                               maxExportBatchSize,
                                               exporterTimeoutNanos,
                                               backpressureStrategy);
  }

  /**
   * Throws an {@link IllegalArgumentException} if the argument is false. This method is similar to
   * {@code Preconditions.checkArgument(boolean, Object)} from Guava.
   *
   * @param isValid      whether the argument check passed.
   * @param errorMessage the message to use for the exception.
   */
  private static void checkArgument(boolean isValid, String errorMessage) {
    if (!isValid) {
      throw new IllegalArgumentException(errorMessage);
    }
  }
}
