/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.logging.otel.configuration.impl.log4j;

import static org.mule.runtime.logging.otel.impl.export.log4j.OpenTelemetryLog4JBridge.addOpenTelemetryLogging;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_CONFIGURATION_FILE_NAME;
import static org.mule.runtime.logging.otel.api.configuration.OpenTelemetryLoggingConfigurationProperties.MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_ENABLED;
import static org.mule.runtime.logging.otel.impl.export.log4j.OpenTelemetryLog4JBridge.OPEN_TELEMETRY_APPENDER_NAME_SUFFIX;

import static java.lang.Boolean.parseBoolean;

import static org.hamcrest.CoreMatchers.notNullValue;
import static org.hamcrest.CoreMatchers.nullValue;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

import org.mule.runtime.api.util.MuleSystemProperties;
import org.mule.runtime.logging.otel.impl.configuration.LogRecordExporterConfiguratorException;
import org.mule.runtime.module.artifact.api.descriptor.DeployableArtifactDescriptor;
import org.mule.runtime.utils.parameters.SystemProperty;

import org.apache.logging.log4j.core.Appender;
import org.apache.logging.log4j.core.LoggerContext;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.RegisterExtension;
import org.junit.jupiter.params.ParameterizedClass;
import org.junit.jupiter.params.provider.CsvSource;

@ParameterizedClass(name = "[{index}] {arguments}")
@CsvSource(useHeadersInDisplayName = true, value = {
    "enableOTELLogging, loggingConfigurationFile",
    "false, null",
    "true, null"
})
class OpenTelemetryLog4JBridgeTestCase {

  private final LoggerContext loggerContextStub = LoggerContext.getContext(false);

  @RegisterExtension
  public final SystemProperty enableOTELLogging;

  @RegisterExtension
  public final SystemProperty loggingConfigurationFile;

  public DeployableArtifactDescriptor artifactDescriptor;

  @RegisterExtension
  public final SystemProperty enableConfigurationAtApplicationLevel =
      new SystemProperty(MuleSystemProperties.ENABLE_OBSERVABILITY_CONFIGURATION_AT_APPLICATION_LEVEL_PROPERTY, "true");

  public OpenTelemetryLog4JBridgeTestCase(String enableOTELLogging, String loggingConfigurationFile) {
    this.enableOTELLogging = new SystemProperty(MULE_OPEN_TELEMETRY_LOGGING_EXPORTER_ENABLED, enableOTELLogging);
    this.loggingConfigurationFile =
        new SystemProperty(MULE_OPEN_TELEMETRY_LOGGING_CONFIGURATION_FILE_NAME, loggingConfigurationFile);
  }

  @BeforeEach
  void setUp() {
    artifactDescriptor = mock(DeployableArtifactDescriptor.class);
    when(artifactDescriptor.getName()).thenReturn("test-artifact");
  }

  @Test
  void testLog4jBridgeWithDefaultConfiguration() throws LogRecordExporterConfiguratorException {
    addOpenTelemetryLogging(loggerContextStub, artifactDescriptor);
    Appender openTelemetryAppender = loggerContextStub.getRootLogger().getAppenders()
        .get("test-artifact" + OPEN_TELEMETRY_APPENDER_NAME_SUFFIX);
    if (parseBoolean(enableOTELLogging.getPropertyValue())) {
      assertThat(openTelemetryAppender, notNullValue());
    } else {
      assertThat(openTelemetryAppender, nullValue());
    }
  }
}
