/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.extension.internal.runtime.transaction;

import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.notNullValue;
import static org.hamcrest.CoreMatchers.nullValue;
import static org.hamcrest.CoreMatchers.sameInstance;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.junit.jupiter.api.Assertions.assertThrows;
import static org.mockito.Mockito.doThrow;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.verify;

import org.mule.runtime.api.connection.ConnectionHandler;
import org.mule.runtime.api.notification.NotificationDispatcher;
import org.mule.runtime.api.tx.TransactionException;
import org.mule.runtime.core.internal.transaction.xa.IllegalTransactionStateException;
import org.mule.runtime.extension.api.runtime.config.ConfigurationInstance;
import org.mule.sdk.api.connectivity.TransactionalConnection;
import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.size.SmallTest;

import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;

@SmallTest
@ExtendWith(MockitoExtension.class)
class ExtensionTransactionTestCase extends AbstractMuleTestCase {

  private static final String APPLICATION_NAME = "testApp";

  @Mock
  private NotificationDispatcher notificationDispatcher;

  @Mock
  private ConfigurationInstance configurationInstance;

  @Mock
  private ConfigurationInstance anotherConfigurationInstance;

  @Mock
  private TransactionalConnection transactionalConnection;

  @Mock
  private ConnectionHandler<TransactionalConnection> connectionHandler;

  @Mock
  private ExtensionTransactionalResource<TransactionalConnection> extensionResource;

  private ExtensionTransaction transaction;
  private ExtensionTransactionKey transactionKey;

  @BeforeEach
  public void setUp() {
    transaction = new ExtensionTransaction(APPLICATION_NAME, notificationDispatcher);
    transactionKey = new ExtensionTransactionKey(configurationInstance);
  }

  @Test
  void constructor() {
    assertThat(transaction, is(notNullValue()));
  }

  @Test
  void getKeyType() {
    assertThat(transaction.getKeyType(), is(sameInstance(ExtensionTransactionKey.class)));
  }

  @Test
  void getResourceType() {
    assertThat(transaction.getResourceType(), is(sameInstance(ExtensionTransactionalResource.class)));
  }

  @Test
  void bindResourceWithValidKeyAndResource() throws Exception {
    transaction.bindResource(transactionKey, extensionResource);

    assertThat(transaction.hasResource(transactionKey), is(true));
    assertThat(transaction.getResource(transactionKey), is(sameInstance(extensionResource)));
    verify(extensionResource).begin();
  }

  @Test
  void bindResourceWithInvalidKey() {
    String invalidKey = "invalidKey";

    assertThrows(IllegalTransactionStateException.class, () -> {
      transaction.bindResource(invalidKey, extensionResource);
    });
  }

  @Test
  void bindResourceWithInvalidResource() {
    String invalidResource = "invalidResource";

    assertThrows(IllegalTransactionStateException.class, () -> {
      transaction.bindResource(transactionKey, invalidResource);
    });
  }

  @Test
  void bindResourceWithInvalidKeyAndResource() {
    String invalidKey = "invalidKey";
    String invalidResource = "invalidResource";

    assertThrows(IllegalTransactionStateException.class, () -> {
      transaction.bindResource(invalidKey, invalidResource);
    });
  }

  @Test
  void bindResourceCallsBegin() throws Exception {
    transaction.bindResource(transactionKey, extensionResource);

    verify(extensionResource).begin();
  }

  @Test
  void bindResourceBeginThrowsException() throws Exception {
    doThrow(new RuntimeException("begin failed")).when(extensionResource).begin();

    assertThrows(TransactionException.class, () -> {
      transaction.bindResource(transactionKey, extensionResource);
    });
  }

  @Test
  void hasResourceWithMatchingKey() throws Exception {
    transaction.bindResource(transactionKey, extensionResource);

    assertThat(transaction.hasResource(transactionKey), is(true));
  }

  @Test
  void hasResourceWithNonMatchingKey() throws Exception {
    transaction.bindResource(transactionKey, extensionResource);

    ExtensionTransactionKey differentKey = new ExtensionTransactionKey(anotherConfigurationInstance);

    assertThat(transaction.hasResource(differentKey), is(false));
  }

  @Test
  void hasResourceWhenNoResourceBound() {
    assertThat(transaction.hasResource(transactionKey), is(false));
  }

  @Test
  void getResourceWithMatchingKey() throws Exception {
    transaction.bindResource(transactionKey, extensionResource);

    assertThat(transaction.getResource(transactionKey), is(sameInstance(extensionResource)));
  }

  @Test
  void getResourceWithNonMatchingKey() throws Exception {
    transaction.bindResource(transactionKey, extensionResource);

    ExtensionTransactionKey differentKey = new ExtensionTransactionKey(anotherConfigurationInstance);

    assertThat(transaction.getResource(differentKey), is(nullValue()));
  }

  @Test
  void getResourceWhenNoResourceBound() {
    assertThat(transaction.getResource(transactionKey), is(nullValue()));
  }

  @Test
  void doCommitWithBoundResource() throws Exception {
    transaction.bindResource(transactionKey, extensionResource);
    transaction.commit();

    verify(extensionResource).commit();
  }

  @Test
  void doCommitWithoutBoundResource() throws Exception {
    // Should not throw exception when no resource is bound
    transaction.commit();

    verify(extensionResource, never()).commit();
  }

  @Test
  void doCommitThrowsException() throws Exception {
    transaction.bindResource(transactionKey, extensionResource);
    doThrow(new RuntimeException("commit failed")).when(extensionResource).commit();

    assertThrows(TransactionException.class, () -> {
      transaction.commit();
    });
  }

  @Test
  void doRollbackWithBoundResource() throws Exception {
    transaction.bindResource(transactionKey, extensionResource);
    transaction.rollback();

    verify(extensionResource).rollback();
  }

  @Test
  void doRollbackWithoutBoundResource() throws Exception {
    // Should not throw exception when no resource is bound
    transaction.rollback();

    verify(extensionResource, never()).rollback();
  }

  @Test
  void doRollbackThrowsException() throws Exception {
    transaction.bindResource(transactionKey, extensionResource);
    doThrow(new RuntimeException("rollback failed")).when(extensionResource).rollback();

    assertThrows(TransactionException.class, () -> {
      transaction.rollback();
    });
  }

  @Test
  void doBeginWithBoundResource() throws Exception {
    transaction.bindResource(transactionKey, extensionResource);

    // begin is called as part of bindResource
    verify(extensionResource).begin();
  }
}

