/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.extension.internal.runtime.transaction;

import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.notNullValue;
import static org.hamcrest.CoreMatchers.sameInstance;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.junit.jupiter.api.Assertions.assertThrows;
import static org.mockito.Mockito.doThrow;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

import org.mule.runtime.api.connection.ConnectionException;
import org.mule.runtime.api.connection.ConnectionProvider;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.tx.TransactionException;
import org.mule.runtime.core.api.transaction.Transaction;
import org.mule.runtime.core.api.transaction.TransactionCoordination;
import org.mule.runtime.core.internal.connection.ConnectionHandlerAdapter;
import org.mule.sdk.api.connectivity.TransactionalConnection;
import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.size.SmallTest;

import org.junit.jupiter.api.AfterEach;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;

@SmallTest
@ExtendWith(MockitoExtension.class)
class TransactionalConnectionHandlerTestCase extends AbstractMuleTestCase {

  @Mock
  private ExtensionTransactionalResource<TransactionalConnection> resource;

  @Mock
  private ConnectionHandlerAdapter<TransactionalConnection> connectionHandler;

  @Mock
  private TransactionalConnection connection;

  @Mock
  private ConnectionProvider<TransactionalConnection> connectionProvider;

  @Mock
  private Transaction transaction;

  private TransactionalConnectionHandler<TransactionalConnection> handler;

  @BeforeEach
  public void setUp() {
    when(resource.getConnectionHandler()).thenReturn(connectionHandler);
    handler = new TransactionalConnectionHandler<>(resource);
  }

  @AfterEach
  public void tearDown() throws Exception {
    Transaction currentTransaction = TransactionCoordination.getInstance().getTransaction();
    if (currentTransaction != null) {
      TransactionCoordination.getInstance().unbindTransaction(currentTransaction);
    }
  }

  @Test
  void constructor() {
    assertThat(handler, is(notNullValue()));
  }

  @Test
  void constructorWithNullResource() {
    assertThrows(NullPointerException.class, () -> {
      new TransactionalConnectionHandler<>(null);
    });
  }

  @Test
  void constructorWithNullConnectionHandler() {
    when(resource.getConnectionHandler()).thenReturn(null);

    assertThrows(NullPointerException.class, () -> {
      new TransactionalConnectionHandler<>(resource);
    });
  }

  @Test
  void getConnection() throws ConnectionException {
    when(resource.getConnection()).thenReturn(connection);

    TransactionalConnection result = handler.getConnection();

    assertThat(result, is(sameInstance(connection)));
    verify(resource).getConnection();
  }

  @Test
  void getConnectionThrowsException() throws ConnectionException {
    when(resource.getConnection()).thenThrow(new RuntimeException("Connection failed"));

    assertThrows(RuntimeException.class, () -> {
      handler.getConnection();
    });
  }

  @Test
  void releaseDoesNothing() {
    // Release should not do anything because connection shouldn't be released until transaction is resolved
    handler.release();

    verify(resource, never()).getConnection();
    verify(connectionHandler, never()).release();
  }

  @Test
  void invalidateWithActiveTransaction() throws Exception {
    TransactionCoordination.getInstance().bindTransaction(transaction);

    handler.invalidate();

    verify(connectionHandler).invalidate();
  }

  @Test
  void invalidateWithNoTransaction() {
    handler.invalidate();

    verify(connectionHandler).invalidate();
  }

  @Test
  void invalidateWithRollbackFailure() throws Exception {
    TransactionCoordination.getInstance().bindTransaction(transaction);
    doThrow(new RuntimeException("Rollback failed")).when(transaction).rollback();

    // Should still invalidate the connection handler even if rollback fails
    handler.invalidate();

    verify(connectionHandler).invalidate();
  }

  @Test
  void closeWithActiveTransaction() throws Exception {
    TransactionCoordination.getInstance().bindTransaction(transaction);

    handler.close();

    verify(connectionHandler).close();
  }

  @Test
  void closeWithNoTransaction() throws Exception {
    handler.close();

    verify(connectionHandler).close();
  }

  @Test
  void closeWithRollbackFailure() throws Exception {
    TransactionCoordination.getInstance().bindTransaction(transaction);
    doThrow(new RuntimeException("Rollback failed")).when(transaction).rollback();

    assertThrows(TransactionException.class, () -> {
      handler.close();
    });

    verify(connectionHandler).close();
  }

  @Test
  void closeWithConnectionHandlerFailure() throws Exception {
    MuleException muleException = new MuleException() {};
    doThrow(muleException).when(connectionHandler).close();

    MuleException thrown = assertThrows(MuleException.class, () -> {
      handler.close();
    });

    assertThat(thrown, is(sameInstance(muleException)));
  }

  @Test
  void closeWithBothRollbackAndConnectionHandlerFailure() throws Exception {
    TransactionCoordination.getInstance().bindTransaction(transaction);
    doThrow(new RuntimeException("Rollback failed")).when(transaction).rollback();
    MuleException muleException = new MuleException() {};
    doThrow(muleException).when(connectionHandler).close();

    // When both fail, the exception from the finally block (connectionHandler.close()) is thrown
    MuleException thrown = assertThrows(MuleException.class, () -> {
      handler.close();
    });

    assertThat(thrown, is(sameInstance(muleException)));
    verify(connectionHandler).close();
  }

  @Test
  void getConnectionProvider() {
    when(connectionHandler.getConnectionProvider()).thenReturn(connectionProvider);

    ConnectionProvider<TransactionalConnection> result = handler.getConnectionProvider();

    assertThat(result, is(sameInstance(connectionProvider)));
    verify(connectionHandler).getConnectionProvider();
  }

  @Test
  void multipleGetConnectionCalls() throws ConnectionException {
    when(resource.getConnection()).thenReturn(connection);

    TransactionalConnection result1 = handler.getConnection();
    TransactionalConnection result2 = handler.getConnection();

    assertThat(result1, is(sameInstance(connection)));
    assertThat(result2, is(sameInstance(connection)));
    verify(resource, times(2)).getConnection();
  }

  @Test
  void multipleReleaseCalls() {
    // Multiple release calls should all do nothing
    handler.release();
    handler.release();
    handler.release();

    verify(connectionHandler, never()).release();
  }
}

