/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.extension.internal.loader.validator;

import static org.mule.runtime.extension.api.util.NameUtils.getComponentModelTypeName;
import static org.mule.runtime.extension.api.util.NameUtils.getModelName;
import static org.mule.runtime.module.extension.internal.value.ValueProviderUtils.getParameterNameFromExtractionExpression;

import static java.lang.String.format;
import static java.lang.String.join;
import static java.util.stream.Collectors.toMap;

import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.StringType;
import org.mule.runtime.api.meta.model.ConnectableComponentModel;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.config.ConfigurationModel;
import org.mule.runtime.api.meta.model.connection.ConnectionProviderModel;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.parameter.ActingParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.meta.model.parameter.ValueProviderModel;
import org.mule.runtime.api.meta.model.source.SourceModel;
import org.mule.runtime.api.meta.model.util.IdempotentExtensionWalker;
import org.mule.runtime.extension.api.loader.ExtensionModelValidator;
import org.mule.runtime.extension.api.loader.Problem;
import org.mule.runtime.extension.api.loader.ProblemsReporter;
import org.mule.runtime.extension.api.util.NameUtils;
import org.mule.runtime.module.extension.internal.loader.java.property.FieldsValueProviderFactoryModelProperty;
import org.mule.runtime.module.extension.internal.loader.java.property.ValueProviderFactoryModelProperty;

import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

/**
 * {@link ExtensionModelValidator} for the correct usage of {@link ValueProviderModel} and
 * {@link ValueProviderFactoryModelProperty}
 *
 * @since 4.10
 */
public abstract class AbstractValueProviderModelValidator<C extends AbstractValueProviderModelValidator.ValidationContext>
    implements ExtensionModelValidator {

  public void doValidate(ExtensionModel model, ProblemsReporter problemsReporter, C validationContext) {
    new IdempotentExtensionWalker() {

      @Override
      protected void onConfiguration(ConfigurationModel model) {
        doValidateModel(model, problemsReporter, validationContext, false);
      }

      @Override
      protected void onConnectionProvider(ConnectionProviderModel model) {
        doValidateModel(model, problemsReporter, validationContext, false);
      }

      @Override
      protected void onSource(SourceModel model) {
        doValidateModel(model, problemsReporter, validationContext, true);
      }

      @Override
      protected void onOperation(OperationModel model) {
        doValidateModel(model, problemsReporter, validationContext, true);
      }
    }.walk(model);
  }

  private void doValidateModel(ParameterizedModel model, ProblemsReporter problemsReporter, C validationContext,
                               boolean supportsConnectionsAndConfigs) {
    model.getAllParameterModels()
        .forEach(param -> {
          Optional<ValueProviderFactoryModelProperty> valueProviderFactoryModelProperty =
              param.getModelProperty(ValueProviderFactoryModelProperty.class);
          Optional<FieldsValueProviderFactoryModelProperty> fieldValueProviderFactoryModelProperty =
              param.getModelProperty(FieldsValueProviderFactoryModelProperty.class);

          if (valueProviderFactoryModelProperty.isPresent() && fieldValueProviderFactoryModelProperty.isPresent()) {
            problemsReporter
                .addError(new Problem(model,
                                      format("Parameter [%s] from %s with name %s has both a Value Provider and a Field Value Provider",
                                             param.getName(), getComponentModelTypeName(model), getModelName(model))));
          } else if (valueProviderFactoryModelProperty.isPresent()) {
            validateOptionsResolver(param, true, null, valueProviderFactoryModelProperty.get(), model, problemsReporter,
                                    validationContext,
                                    supportsConnectionsAndConfigs);
          } else {
            fieldValueProviderFactoryModelProperty.map(FieldsValueProviderFactoryModelProperty::getFieldsValueProviderFactories)
                .ifPresent(factories -> factories
                    .forEach((targetSelector,
                              fieldsValueProviderFactoryModelProperty) -> validateOptionsResolver(param, false,
                                                                                                  targetSelector,
                                                                                                  fieldsValueProviderFactoryModelProperty,
                                                                                                  model, problemsReporter,
                                                                                                  validationContext,
                                                                                                  supportsConnectionsAndConfigs)));
          }
        });
  }

  private void validateOptionsResolver(ParameterModel param, boolean mustBeStringType,
                                       String targetSelector,
                                       ValueProviderFactoryModelProperty modelProperty,
                                       ParameterizedModel model, ProblemsReporter problemsReporter,
                                       C validationContext,
                                       boolean supportsConnectionsAndConfigs) {
    Optional<? extends ValueProviderModel> valueProviderModel;
    if (targetSelector != null) {
      valueProviderModel = param.getFieldValueProviderModels().stream()
          .filter(fieldValueProviderModel -> fieldValueProviderModel.getTargetSelector().equals(targetSelector))
          .findAny();
    } else {
      valueProviderModel = param.getValueProviderModel();
    }

    if (valueProviderModel.isEmpty()) {
      throw new IllegalStateException(format("Parameter %s from %s with name %s has should have a ValueProviderModel associated.",
                                             param.getName(), getComponentModelTypeName(model), getModelName(model)));
    }
    List<ParameterModel> allContainerParameters = model.getAllParameterModels();

    doValidateProvider(model, modelProperty, valueProviderModel.get(), problemsReporter, validationContext);

    if (validateComponentHasParametersWithRepeatedNames(allContainerParameters, param, problemsReporter, model)) {
      return;
    }

    Map<String, MetadataType> containerParameterTypesByName =
        allContainerParameters.stream().collect(toMap(ParameterModel::getName, ParameterModel::getType));
    String modelName = NameUtils.getModelName(model);
    String modelTypeName = getComponentModelTypeName(model);

    if (mustBeStringType && !(param.getType() instanceof StringType)) {
      problemsReporter.addError(new Problem(model,
                                            format("The parameter [%s] of the %s '%s' is not of String type. Parameters that provides Values should be of String type.",
                                                   param.getName(), modelTypeName, modelName)));
    }

    String providerId = getProviderId(modelProperty, valueProviderModel.get());
    for (ActingParameterModel actingParameterModel : modelProperty.getActingParameterModels()) {
      String parameterNameFromExtractionExpression =
          getParameterNameFromExtractionExpression(actingParameterModel.getExtractionExpression());
      if (!containerParameterTypesByName.containsKey(parameterNameFromExtractionExpression)) {
        problemsReporter.addError(new Problem(model,
                                              format("The Value Provider [%s] declares to use a parameter '%s' which doesn't exist in the %s '%s'",
                                                     providerId, parameterNameFromExtractionExpression, modelTypeName,
                                                     modelName)));
      }
    }

    doValidateInjectableParameters(model, modelName, modelTypeName, containerParameterTypesByName, modelProperty,
                                   valueProviderModel.get(), problemsReporter, validationContext);
    reportProblems(model, problemsReporter, supportsConnectionsAndConfigs, valueProviderModel, modelName, modelTypeName,
                   providerId);
  }

  private void reportProblems(ParameterizedModel model, ProblemsReporter problemsReporter, boolean supportsConnectionsAndConfigs,
                              Optional<? extends ValueProviderModel> valueProviderModel, String modelName, String modelTypeName,
                              String providerId) {
    boolean usesConnection = valueProviderModel.get().requiresConnection();
    boolean usesConfig = valueProviderModel.get().requiresConfiguration();
    if (supportsConnectionsAndConfigs && usesConnection
        && model instanceof ConnectableComponentModel connectableComponentModel
        && connectableComponentModel.requiresConnection() != usesConnection) {
      problemsReporter.addError(new Problem(model,
                                            format("The Value Provider [%s] defines that requires a connection, but is used in the %s '%s' which is connection less",
                                                   providerId, modelTypeName, modelName)));
    }

    if (!supportsConnectionsAndConfigs) {
      if (usesConnection) {
        problemsReporter.addError(new Problem(model,
                                              format("The Value Provider [%s] defines that requires a connection which is not allowed for a Value Provider of a %s's parameter [%s]",
                                                     providerId, modelTypeName, modelName)));
      }

      if (usesConfig) {
        problemsReporter.addError(new Problem(model,
                                              format("The Value Provider [%s] defines that requires a configuration which is not allowed for a Value Provider of a %s's parameter [%s]",
                                                     providerId, modelTypeName, modelName)));
      }
    }
  }

  private boolean validateComponentHasParametersWithRepeatedNames(List<ParameterModel> allParameterModels, ParameterModel param,
                                                                  ProblemsReporter problemsReporter,
                                                                  ParameterizedModel parameterizedModel) {
    Set<String> repeatedParameterNames = new HashSet<>();
    Set<String> parameterNames = new HashSet<>();
    for (ParameterModel parameterModel : allParameterModels) {
      if (parameterNames.contains(parameterModel.getName())) {
        repeatedParameterNames.add(parameterModel.getName());
      } else {
        parameterNames.add(parameterModel.getName());
      }
    }
    if (!repeatedParameterNames.isEmpty()) {
      problemsReporter
          .addError(new Problem(param,
                                format("Parameter [%s] from %s with name %s has a Value Provider defined, but that %s has one or more parameters with repeated names [%s]. Components with parameters with non-unique names do not support Value Providers",
                                       param.getName(), getComponentModelTypeName(parameterizedModel),
                                       parameterizedModel.getName(), getComponentModelTypeName(parameterizedModel),
                                       join(", ", repeatedParameterNames))));
      return true;
    }
    return false;
  }

  protected String getProviderId(ValueProviderFactoryModelProperty modelProperty, ValueProviderModel valueProviderModel) {
    return valueProviderModel.getProviderId();
  }

  protected void doValidateProvider(ParameterizedModel containerModel, ValueProviderFactoryModelProperty modelProperty,
                                    ValueProviderModel valueProviderModel, ProblemsReporter problemsReporter,
                                    C validationContext) {
    // Does nothing
  }

  protected void doValidateInjectableParameters(ParameterizedModel containerModel,
                                                String containerName, String containerTypeName,
                                                Map<String, MetadataType> containerParameterTypesByName,
                                                ValueProviderFactoryModelProperty modelProperty,
                                                ValueProviderModel valueProviderModel,
                                                ProblemsReporter problemsReporter,
                                                C validationContext) {
    // Does nothing
  }

  public interface ValidationContext {

  }
}
