/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.extension.internal.loader.parser.java;

import static java.util.Optional.empty;
import static java.util.Optional.of;
import static java.util.stream.Collectors.toMap;

import org.mule.metadata.api.ClassTypeLoader;
import org.mule.runtime.extension.api.loader.parser.FieldValueProviderModelParser;
import org.mule.runtime.module.extension.api.loader.java.type.AnnotationValueFetcher;
import org.mule.runtime.module.extension.api.loader.java.type.ExtensionParameter;
import org.mule.sdk.api.annotation.binding.Binding;
import org.mule.sdk.api.annotation.values.FieldValues;
import org.mule.sdk.api.annotation.values.FieldsValues;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;

/**
 * {@link FieldValueProviderModelParser} for Java based syntax
 *
 * @since 4.10.0
 * @implNote Logic was extracted from what used to be the ValueProvidersParameterDeclarationEnricher in earlier versions.
 */
public class JavaFieldValueProviderModelParser extends JavaValueProviderModelParser implements FieldValueProviderModelParser {

  public static List<FieldValueProviderModelParser> getParserFromFieldValues(ExtensionParameter parameter,
                                                                             ClassTypeLoader typeLoader) {
    List<FieldValueProviderModelParser> fieldValuesList = new ArrayList<>();
    parameter.getValueFromAnnotation(FieldsValues.class)
        .map(fieldsValues -> fieldsValues.getInnerAnnotations(FieldsValues::value))
        .ifPresent(fetchers -> fetchers.stream()
            .map(fetcher -> getParserFromFetcher(fetcher, typeLoader))
            .forEach(mp -> mp.ifPresent(fieldValuesList::add)));
    parameter.getValueFromAnnotation(FieldValues.class)
        .flatMap(fetcher -> getParserFromFetcher(fetcher, typeLoader))
        .ifPresent(fieldValuesList::add);

    return fieldValuesList;
  }

  private static Optional<JavaFieldValueProviderModelParser> getParserFromFetcher(AnnotationValueFetcher<FieldValues> valueFetcher,
                                                                                  ClassTypeLoader typeLoader) {
    Optional<Class<?>> value = valueFetcher.getClassValue(FieldValues::value).getDeclaringClass();
    if (value.isEmpty()) {
      return empty();
    }
    return of(new JavaFieldValueProviderModelParser(valueFetcher.getArrayValue(FieldValues::targetSelectors),
                                                    value.get(),
                                                    valueFetcher.getBooleanValue(FieldValues::open),
                                                    getBindingsMapFromFetcher(valueFetcher
                                                        .getInnerAnnotations(FieldValues::bindings)),
                                                    typeLoader));
  }

  private static Map<String, String> getBindingsMapFromFetcher(List<AnnotationValueFetcher<Binding>> annotationValueFetcher) {
    return annotationValueFetcher.stream()
        .collect(toMap(fetcher -> fetcher.getStringValue(Binding::actingParameter),
                       fetcher -> fetcher.getStringValue(Binding::extractionExpression)));
  }

  private final List<String> targetSelectors;

  public JavaFieldValueProviderModelParser(List<String> targetSelectors,
                                           Class<?> clazz,
                                           boolean isOpen,
                                           Map<String, String> bindingsMap,
                                           ClassTypeLoader typeLoader) {
    super(clazz, isOpen, bindingsMap, empty(), typeLoader, false);
    this.targetSelectors = targetSelectors;
  }

  @Override
  public List<String> getTargetSelectors() {
    return targetSelectors;
  }
}
