/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.extension.internal.value;

import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.initialiseIfNeeded;
import static org.mule.runtime.extension.api.values.ValueResolvingException.UNKNOWN;

import static java.util.Optional.empty;
import static java.util.Optional.of;

import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.meta.model.parameter.ParameterizedModel;
import org.mule.runtime.api.meta.model.parameter.ValueProviderModel;
import org.mule.runtime.api.parameterization.ComponentParameterization;
import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.extension.api.loader.parser.ValueProviderFactory;
import org.mule.runtime.extension.api.loader.parser.ValueProviderFactoryContext;
import org.mule.runtime.extension.api.values.ValueProvider;
import org.mule.runtime.extension.api.values.ValueResolvingException;
import org.mule.runtime.module.extension.api.runtime.resolver.ParameterValueResolver;
import org.mule.runtime.module.extension.internal.runtime.ResolverBasedComponentParameterization;

import java.util.Optional;
import java.util.function.Supplier;

/**
 * Wrapper around a {@link ValueProviderFactory} taking care of generating the parameter resolvers and applying lifecycle to the
 * result.
 *
 * @since 4.10.0
 */
public class ValueProviderFactoryWrapper {

  private final ValueProviderFactory valueProviderFactory;
  private final MuleContext muleContext;
  private final ValueProviderFactoryContext valueProviderFactoryContext;

  public ValueProviderFactoryWrapper(ValueProviderModel model,
                                     ValueProviderFactory valueProviderFactory,
                                     ParameterValueResolver parameterValueResolver,
                                     Supplier<Object> connectionSupplier,
                                     Supplier<Object> configurationSupplier,
                                     MuleContext muleContext,
                                     ParameterizedModel parameterizedModel) {
    this.valueProviderFactory = valueProviderFactory;
    this.muleContext = muleContext;
    ComponentParameterization<?> componentParameterization =
        new ResolverBasedComponentParameterization<>(parameterizedModel, parameterValueResolver);
    this.valueProviderFactoryContext =
        new DefaultValueProviderFactoryContext(model, componentParameterization, connectionSupplier, configurationSupplier);

    // TODO W-19057747: see if we can do this differently
    try {
      muleContext.getInjector().inject(valueProviderFactory);
    } catch (MuleException e) {
      throw new RuntimeException(e);
    }
  }

  ValueProvider createValueProvider() throws ValueResolvingException {
    try {
      ValueProvider valueProvider = valueProviderFactory.create(valueProviderFactoryContext);
      // We need to use the version with MuleContext because these objects can be injected the MuleContext itself, and we need
      // to retain that for compatibility
      initialiseIfNeeded(valueProvider, muleContext);

      return valueProvider;
    } catch (ValueResolvingException e) {
      throw e;
    } catch (Exception e) {
      throw new ValueResolvingException("An error occurred trying to create a ValueProvider", UNKNOWN, e);
    }
  }

  private static class DefaultValueProviderFactoryContext implements ValueProviderFactoryContext {

    private final ValueProviderModel valueProviderModel;
    private final ComponentParameterization<?> componentParameterization;
    private final Supplier<Object> connectionSupplier;
    private final Supplier<Object> configurationSupplier;

    private DefaultValueProviderFactoryContext(ValueProviderModel valueProviderModel,
                                               ComponentParameterization<?> componentParameterization,
                                               Supplier<Object> connectionSupplier, Supplier<Object> configurationSupplier) {
      this.valueProviderModel = valueProviderModel;
      this.componentParameterization = componentParameterization;
      this.connectionSupplier = connectionSupplier;
      this.configurationSupplier = configurationSupplier;
    }

    @Override
    public ValueProviderModel getModel() {
      return valueProviderModel;
    }

    @Override
    public ComponentParameterization<?> getComponentParameterization() {
      return componentParameterization;
    }

    @Override
    public Optional<Supplier<?>> getConfigurationSupplier() {
      return valueProviderModel.requiresConfiguration() ? of(configurationSupplier) : empty();
    }

    @Override
    public Optional<Supplier<?>> getConnectionSupplier() {
      return valueProviderModel.requiresConnection() ? of(connectionSupplier) : empty();
    }
  }
}
