/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.extension.internal.manager;

import static org.mule.runtime.api.util.MuleSystemProperties.SYSTEM_PROPERTY_PREFIX;
import static org.mule.runtime.api.util.Preconditions.checkArgument;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.disposeIfNeeded;
import static org.mule.runtime.core.api.lifecycle.LifecycleUtils.stopIfNeeded;
import static org.mule.runtime.core.internal.util.ClassUtils.withContextClassLoader;
import static org.mule.runtime.core.internal.util.version.JdkVersionUtils.getJdkVersion;
import static org.mule.runtime.extension.api.util.ExtensionModelUtils.requiresConfig;
import static org.mule.runtime.module.extension.internal.manager.DefaultConfigurationExpirationMonitor.Builder.newBuilder;
import static org.mule.runtime.module.extension.internal.util.MuleExtensionUtils.getClassLoader;
import static org.mule.runtime.module.extension.internal.util.MuleExtensionUtils.getImplicitConfigurationProviderName;

import static java.lang.String.format;
import static java.lang.System.getProperty;

import static org.apache.commons.lang3.StringUtils.isBlank;

import org.mule.runtime.api.artifact.Registry;
import org.mule.runtime.api.config.ArtifactEncoding;
import org.mule.runtime.api.config.FeatureFlaggingService;
import org.mule.runtime.api.exception.MuleException;
import org.mule.runtime.api.lifecycle.Initialisable;
import org.mule.runtime.api.lifecycle.InitialisationException;
import org.mule.runtime.api.lifecycle.Startable;
import org.mule.runtime.api.lifecycle.Stoppable;
import org.mule.runtime.api.meta.model.ComponentModel;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.config.ConfigurationModel;
import org.mule.runtime.api.time.Time;
import org.mule.runtime.core.api.MuleContext;
import org.mule.runtime.core.api.event.CoreEvent;
import org.mule.runtime.core.api.extension.ExtensionManager;
import org.mule.runtime.core.internal.registry.DefaultRegistry;
import org.mule.runtime.core.internal.util.version.JdkVersionUtils.JdkVersion;
import org.mule.runtime.extension.api.property.ImplicitConfigNameModelProperty;
import org.mule.runtime.extension.api.property.ManyImplicitConfigsModelProperty;
import org.mule.runtime.extension.api.runtime.config.ConfigurationInstance;
import org.mule.runtime.extension.api.runtime.config.ConfigurationProvider;
import org.mule.runtime.module.extension.internal.manager.jdk.ExtensionJdkValidator;
import org.mule.runtime.module.extension.internal.manager.jdk.LooseExtensionJdkValidator;
import org.mule.runtime.module.extension.internal.manager.jdk.NullExtensionJdkValidator;
import org.mule.runtime.module.extension.internal.manager.jdk.StrictExtensionJdkValidator;
import org.mule.runtime.module.extension.internal.runtime.config.DefaultImplicitConfigurationProviderFactory;
import org.mule.runtime.module.extension.internal.runtime.config.ImplicitConfigurationProviderFactory;
import org.mule.runtime.module.extension.internal.util.ReflectionCache;

import java.util.Optional;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import jakarta.inject.Inject;

/**
 * Default implementation of {@link ExtensionManager}. This implementation uses standard Java SPI as a discovery mechanism.
 * <p/>
 * Although it allows registering {@link ConfigurationProvider} instances through the
 * {@link #registerConfigurationProvider(ConfigurationProvider)} method (and that's still the correct way of registering them),
 * this implementation automatically acknowledges any {@link ConfigurationProvider} already present on the {@link Registry}
 *
 * @since 3.7.0
 */
public final class DefaultExtensionManager implements ExtensionManager, Initialisable, Startable, Stoppable {

  private static final Logger LOGGER = LoggerFactory.getLogger(DefaultExtensionManager.class);

  /**
   * System property to set the enforcement policy. Defined here as a decision was made not to expose it as an API yet. For now,
   * it will be for internal use only.
   *
   * @since 4.5.0
   */
  static final String EXTENSION_JVM_ENFORCEMENT_PROPERTY = SYSTEM_PROPERTY_PREFIX + "jvm.version.extension.enforcement";
  static final String JVM_ENFORCEMENT_STRICT = "STRICT";
  static final String JVM_ENFORCEMENT_LOOSE = "LOOSE";
  static final String JVM_ENFORCEMENT_DISABLED = "DISABLED";

  private ImplicitConfigurationProviderFactory implicitConfigurationProviderFactory;
  private final AtomicBoolean initialised = new AtomicBoolean(false);

  @Inject
  private ReflectionCache reflectionCache;

  @Inject
  private FeatureFlaggingService featureFlaggingService;

  private ArtifactEncoding artifactEncoding;

  @Inject
  private MuleContext muleContext;

  private ExtensionRegistry extensionRegistry;
  private ConfigurationExpirationMonitor configurationExpirationMonitor;

  private ExtensionActivator extensionActivator;
  private ExtensionJdkValidator extensionJdkValidator;

  @Override
  public void initialise() throws InitialisationException {
    if (initialised.compareAndSet(false, true)) {
      extensionRegistry = new ExtensionRegistry(new DefaultRegistry(muleContext));
      extensionActivator = new ExtensionActivator(muleContext);
      implicitConfigurationProviderFactory = new DefaultImplicitConfigurationProviderFactory(artifactEncoding,
                                                                                             muleContext, featureFlaggingService);
      resolveJdkValidator();
    }
  }

  /**
   * Starts the {@link #configurationExpirationMonitor}
   *
   * @throws MuleException if it fails to start
   */
  @Override
  public void start() throws MuleException {
    configurationExpirationMonitor = newConfigurationExpirationMonitor();
    configurationExpirationMonitor.beginMonitoring();
    extensionActivator.start();
  }

  /**
   * Stops the {@link #configurationExpirationMonitor}
   *
   * @throws MuleException if it fails to stop
   */
  @Override
  public void stop() throws MuleException {
    extensionActivator.stop();
    configurationExpirationMonitor.stopMonitoring();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void registerExtension(ExtensionModel extensionModel) {
    final String extensionName = extensionModel.getName();
    final String extensionVersion = extensionModel.getVersion();
    final String extensionVendor = extensionModel.getVendor();

    if (LOGGER.isDebugEnabled()) {
      LOGGER.debug("Registering extension {} (version: {} vendor: {} )", extensionName, extensionVersion, extensionVendor);
    }

    extensionJdkValidator.validateJdkSupport(extensionModel);

    if (extensionRegistry.containsExtension(extensionName)) {
      if (LOGGER.isDebugEnabled()) {
        LOGGER.debug("An extension of name '{}' (version: {} vendor {}) is already registered. Skipping...", extensionName,
                     extensionVersion, extensionVendor);
      }
    } else {
      withContextClassLoader(getClassLoader(extensionModel), () -> {
        extensionRegistry.registerExtension(extensionName, extensionModel);
        extensionActivator.activateExtension(extensionModel);
      });
    }
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void registerConfigurationProvider(ConfigurationProvider configurationProvider) {
    extensionRegistry.registerConfigurationProvider(configurationProvider, muleContext);
  }

  @Override
  public void unregisterConfigurationProvider(ConfigurationProvider configurationProvider) {
    extensionRegistry.unregisterConfigurationProvider(configurationProvider, muleContext);
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public ConfigurationInstance getConfiguration(String configurationProviderName, CoreEvent muleEvent) {
    return getConfigurationProvider(configurationProviderName)
        .map(provider -> provider.get(muleEvent))
        .orElseThrow(() -> new IllegalArgumentException(format("There is no registered configurationProvider under name '%s'",
                                                               configurationProviderName)));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Optional<ConfigurationInstance> getConfiguration(ExtensionModel extensionModel,
                                                          ComponentModel componentModel,
                                                          CoreEvent muleEvent) {

    return getConfigurationProvider(extensionModel, componentModel, muleEvent).map(p -> p.get(muleEvent));
  }

  @Override
  public Optional<ConfigurationProvider> getConfigurationProvider(ExtensionModel extensionModel,
                                                                  ComponentModel componentModel,
                                                                  CoreEvent muleEvent) {

    Optional<ConfigurationProvider> configurationProvider = getConfigurationProvider(extensionModel, componentModel);
    if (configurationProvider.isPresent()) {
      return configurationProvider;
    }

    Optional<ConfigurationModel> configurationModel =
        componentModel.getModelProperty(ImplicitConfigNameModelProperty.class)
            .flatMap(mp -> extensionModel.getConfigurationModel(mp.getImplicitConfigName()));;

    return configurationModel.map(c -> createImplicitConfiguration(extensionModel, c, muleEvent));
  }

  @Override
  public Optional<ConfigurationProvider> getConfigurationProvider(ExtensionModel extensionModel, ComponentModel componentModel) {
    if (componentModel.getModelProperty(ManyImplicitConfigsModelProperty.class).isPresent()) {
      throw new IllegalStateException(format("No configuration can be inferred for extension '%s'", extensionModel.getName()));
    }

    Optional<ConfigurationModel> extensionConfigurationModel =
        componentModel.getModelProperty(ImplicitConfigNameModelProperty.class)
            .flatMap(mp -> extensionModel.getConfigurationModel(mp.getImplicitConfigName()));;

    if (!extensionConfigurationModel.isPresent() && requiresConfig(extensionModel, componentModel)) {
      throw new NoConfigRefFoundException(extensionModel, componentModel);
    }
    return extensionConfigurationModel
        .flatMap(c -> getConfigurationProvider(getImplicitConfigurationProviderName(extensionModel, c,
                                                                                    muleContext.getArtifactType(),
                                                                                    muleContext.getId(),
                                                                                    featureFlaggingService)));
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Optional<ConfigurationProvider> getConfigurationProvider(String configurationProviderName) {
    checkArgument(!isBlank(configurationProviderName), "cannot get configuration from a blank provider name");
    return extensionRegistry.getConfigurationProvider(configurationProviderName);
  }

  private ConfigurationProvider createImplicitConfiguration(ExtensionModel extensionModel,
                                                            ConfigurationModel configurationModel,
                                                            CoreEvent muleEvent) {

    String implicitConfigurationProviderName =
        getImplicitConfigurationProviderName(extensionModel, configurationModel, muleContext.getArtifactType(),
                                             muleContext.getId(), featureFlaggingService);

    return extensionRegistry.getConfigurationProvider(implicitConfigurationProviderName).orElseGet(() -> {
      // For an extension model, its instance is always the same
      // This synchronization is not so that this code doesn't run in parallel, but so that it doesn't run more than once for the
      // same extension.
      synchronized (extensionModel) {
        // check that another thread didn't beat us to create the instance
        return extensionRegistry.getConfigurationProvider(implicitConfigurationProviderName).orElseGet(() -> {
          ConfigurationProvider implicitConfigurationProvider =
              implicitConfigurationProviderFactory.createImplicitConfigurationProvider(extensionModel,
                                                                                       configurationModel,
                                                                                       muleEvent,
                                                                                       getReflectionCache(),
                                                                                       muleContext.getExpressionManager());
          registerConfigurationProvider(implicitConfigurationProvider);
          return implicitConfigurationProvider;
        });
      }
    });
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Set<ExtensionModel> getExtensions() {
    return extensionRegistry.getExtensions();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public Optional<ExtensionModel> getExtension(String extensionName) {
    return extensionRegistry.getExtension(extensionName);
  }

  private ConfigurationExpirationMonitor newConfigurationExpirationMonitor() {
    Time freq = getConfigurationExpirationFrequency();
    return newBuilder(extensionRegistry, muleContext).runEvery(freq.getTime(), freq.getUnit())
        .onExpired((key, object) -> disposeConfiguration(key, object)).build();
  }

  @Override
  public void disposeConfiguration(String key, ConfigurationInstance configuration) {
    try {
      stopIfNeeded(configuration);
      disposeIfNeeded(configuration, LOGGER);
    } catch (Exception e) {
      LOGGER.error(format("Could not dispose expired dynamic config of key '%s' and type %s", key,
                          configuration.getClass().getName()),
                   e);
    }
  }

  private Time getConfigurationExpirationFrequency() {
    return muleContext.getConfiguration().getDynamicConfigExpiration().getFrequency();
  }

  void resolveJdkValidator() {
    JdkVersion runningJdkVersion = getJdkVersion();
    String enforcementMode = getProperty(EXTENSION_JVM_ENFORCEMENT_PROPERTY, JVM_ENFORCEMENT_STRICT);

    if (JVM_ENFORCEMENT_STRICT.equals(enforcementMode)) {
      extensionJdkValidator = new StrictExtensionJdkValidator(runningJdkVersion);
    } else if (JVM_ENFORCEMENT_LOOSE.equals(enforcementMode)) {
      extensionJdkValidator = new LooseExtensionJdkValidator(runningJdkVersion, LOGGER);
    } else if (JVM_ENFORCEMENT_DISABLED.equals(enforcementMode)) {
      extensionJdkValidator = new NullExtensionJdkValidator(runningJdkVersion);
    } else {
      throw new IllegalArgumentException("Unsupported " + EXTENSION_JVM_ENFORCEMENT_PROPERTY + "value: " + enforcementMode);
    }
  }

  ExtensionJdkValidator getExtensionJdkValidator() {
    return extensionJdkValidator;
  }

  public ReflectionCache getReflectionCache() {
    if (reflectionCache == null) {
      reflectionCache = new ReflectionCache();
    }
    return reflectionCache;
  }

  @Inject
  public void setArtifactEncoding(ArtifactEncoding artifactEncoding) {
    this.artifactEncoding = artifactEncoding;
  }
}
