/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.extension.internal.runtime.client;

import static java.util.concurrent.ForkJoinPool.commonPool;
import static java.util.concurrent.TimeUnit.SECONDS;

import java.util.Collection;
import java.util.List;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Future;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicInteger;

import com.github.benmanes.caffeine.cache.LoadingCache;

/**
 * Executor meant to be used in {@link LoadingCache} disposal operations, as explained in
 * {@code https://github.com/ben-manes/caffeine/issues/104#issuecomment-238068997}
 *
 * @since 4.5.0
 */
final class ShutdownExecutor implements ExecutorService {

  private final AtomicInteger tasks = new AtomicInteger();
  private final Semaphore semaphore = new Semaphore(0);
  private final ExecutorService delegate = commonPool();
  private volatile boolean shutdown;

  @Override
  public void execute(Runnable task) {
    if (shutdown) {
      throw new RejectedExecutionException("Shutdown");
    }

    tasks.incrementAndGet();
    delegate.execute(() -> {
      try {
        task.run();
      } finally {
        semaphore.release();
      }
    });
  }

  @Override
  public void shutdown() {
    shutdown = true;
  }

  @Override
  public List<Runnable> shutdownNow() {
    return delegate.shutdownNow();
  }

  @Override
  public boolean isShutdown() {
    return delegate.isShutdown();
  }

  @Override
  public boolean awaitTermination(long timeout, TimeUnit unit) throws InterruptedException {
    int permits = tasks.get();
    boolean terminated = semaphore.tryAcquire(permits, timeout, unit);
    if (terminated) {
      semaphore.release(permits);
    }
    return terminated && shutdown;
  }

  @Override
  public boolean isTerminated() {
    try {
      return awaitTermination(0, SECONDS);
    } catch (InterruptedException e) {
      return false;
    }
  }

  @Override
  public <T> Future<T> submit(Callable<T> task) {
    return delegate.submit(task);
  }

  @Override
  public <T> Future<T> submit(Runnable task, T result) {
    return delegate.submit(task, result);
  }

  @Override
  public Future<?> submit(Runnable task) {
    return delegate.submit(task);
  }

  @Override
  public <T> List<Future<T>> invokeAll(Collection<? extends Callable<T>> tasks) throws InterruptedException {
    return delegate.invokeAll(tasks);
  }

  @Override
  public <T> List<Future<T>> invokeAll(Collection<? extends Callable<T>> tasks, long timeout, TimeUnit unit)
      throws InterruptedException {
    return delegate.invokeAll(tasks, timeout, unit);
  }

  @Override
  public <T> T invokeAny(Collection<? extends Callable<T>> tasks) throws InterruptedException, ExecutionException {
    return delegate.invokeAny(tasks);
  }

  @Override
  public <T> T invokeAny(Collection<? extends Callable<T>> tasks, long timeout, TimeUnit unit)
      throws InterruptedException, ExecutionException, TimeoutException {
    return delegate.invokeAny(tasks, timeout, unit);
  }
}
