/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.extension.internal.loader.validation;

import static org.mule.test.module.extension.internal.util.ExtensionsTestUtils.visitableMock;

import static java.util.Arrays.asList;
import static java.util.Collections.emptyList;
import static java.util.Collections.emptySet;
import static java.util.Collections.singletonList;
import static java.util.Collections.singletonMap;
import static java.util.Optional.empty;
import static java.util.Optional.of;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.collection.IsIterableContainingInOrder.contains;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;
import static org.mockito.quality.Strictness.LENIENT;

import org.mule.metadata.api.annotation.EnumAnnotation;
import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.impl.DefaultArrayType;
import org.mule.metadata.api.model.impl.DefaultNumberType;
import org.mule.metadata.api.model.impl.DefaultObjectType;
import org.mule.metadata.api.model.impl.DefaultStringType;
import org.mule.metadata.java.api.JavaTypeLoader;
import org.mule.metadata.java.api.annotation.ClassInformationAnnotation;
import org.mule.runtime.api.meta.model.ExtensionModel;
import org.mule.runtime.api.meta.model.config.ConfigurationModel;
import org.mule.runtime.api.meta.model.operation.OperationModel;
import org.mule.runtime.api.meta.model.parameter.ActingParameterModel;
import org.mule.runtime.api.meta.model.parameter.FieldValueProviderModel;
import org.mule.runtime.api.meta.model.parameter.ParameterGroupModel;
import org.mule.runtime.api.meta.model.parameter.ParameterModel;
import org.mule.runtime.api.meta.model.parameter.ValueProviderModel;
import org.mule.runtime.api.util.Pair;
import org.mule.runtime.api.value.Value;
import org.mule.runtime.extension.api.annotation.param.Connection;
import org.mule.runtime.extension.api.loader.Problem;
import org.mule.runtime.extension.api.loader.ProblemsReporter;
import org.mule.runtime.extension.api.loader.parser.ValueProviderFactory;
import org.mule.runtime.extension.api.model.parameter.ImmutableActingParameterModel;
import org.mule.runtime.extension.api.values.ValueProvider;
import org.mule.runtime.extension.api.values.ValueResolvingException;
import org.mule.runtime.module.extension.api.loader.java.type.FieldElement;
import org.mule.runtime.module.extension.internal.loader.java.property.DeclaringMemberModelProperty;
import org.mule.runtime.module.extension.internal.loader.java.property.FieldsValueProviderFactoryModelProperty;
import org.mule.runtime.module.extension.internal.loader.java.property.ImplementingParameterModelProperty;
import org.mule.runtime.module.extension.internal.loader.java.property.ValueProviderFactoryModelProperty;
import org.mule.runtime.module.extension.internal.loader.java.type.runtime.FieldWrapper;
import org.mule.runtime.module.extension.internal.loader.java.validation.JavaValueProviderModelValidator;
import org.mule.runtime.module.extension.internal.loader.parser.java.JavaValueProviderFactory;
import org.mule.sdk.api.annotation.param.Config;
import org.mule.sdk.api.annotation.param.Parameter;
import org.mule.tck.size.SmallTest;

import java.io.InputStream;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.hamcrest.collection.IsEmptyCollection;
import org.junit.jupiter.api.BeforeEach;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.extension.ExtendWith;
import org.mockito.Mock;
import org.mockito.junit.jupiter.MockitoExtension;
import org.mockito.junit.jupiter.MockitoSettings;

@SmallTest
@ExtendWith(MockitoExtension.class)
@MockitoSettings(strictness = LENIENT)
public class JavaValueProviderModelValidatorTestCase {

  private final JavaTypeLoader loader = new JavaTypeLoader(this.getClass().getClassLoader());
  private final MetadataType STRING_TYPE = loader.load(String.class);
  private final MetadataType NUMBER_TYPE = loader.load(Integer.class);
  private final MetadataType OBJECT_TYPE = loader.load(InputStream.class);
  private final MetadataType STRING_TYPE_WITH_ANNOTATIONS = new DefaultStringType(MetadataFormat.JAVA, createStringAnnotations());
  private final MetadataType NUMBER_TYPE_WITH_ANNOTATIONS = new DefaultNumberType(MetadataFormat.JAVA, createNumberAnnotations());
  private final MetadataType OBJECT_TYPE_WITH_ANNOTATIONS =
      new DefaultObjectType(emptyList(), false, null, MetadataFormat.JSON, createObjectAnnotations());
  private final MetadataType ARRAY_TYPE_WITH_ANNOTATIONS =
      new DefaultArrayType(() -> STRING_TYPE, MetadataFormat.JAVA, createObjectAnnotations());
  private JavaValueProviderModelValidator valueProviderModelValidator;

  private ProblemsReporter problemsReporter;

  @Mock
  ExtensionModel extensionModel;

  @Mock
  OperationModel operationModel;

  @Mock
  ParameterModel operationParameter;

  @Mock
  ParameterModel anotherOperationParameter;

  @Mock
  ParameterModel configurationParameter;

  @Mock
  ConfigurationModel configurationModel;

  @Mock
  ParameterGroupModel parameterGroupModel;

  @Mock
  ParameterGroupModel configurationParameterGroupModel;

  private static final ValueProviderFactoryModelProperty SOME_VALUE_PROVIDER_MP_NO_PARAMS =
      fromValueProviderClass(SomeValueProvider.class);

  @BeforeEach
  public void setUp() {
    valueProviderModelValidator = new JavaValueProviderModelValidator();
    problemsReporter = new ProblemsReporter(extensionModel);

    visitableMock(operationModel);

    when(extensionModel.getConfigurationModels()).thenReturn(singletonList(configurationModel));
    when(configurationModel.getAllParameterModels()).thenReturn(singletonList(configurationParameter));
    when(configurationModel.getParameterGroupModels()).thenReturn(singletonList(configurationParameterGroupModel));
    when(configurationModel.getName()).thenReturn("SomeConfig");
    when(configurationParameterGroupModel.getParameterModels()).thenReturn(singletonList(configurationParameter));

    when(extensionModel.getOperationModels()).thenReturn(singletonList(operationModel));
    when(operationModel.getAllParameterModels()).thenReturn(asList(operationParameter, anotherOperationParameter));
    when(operationModel.getName()).thenReturn("superOperation");
    when(parameterGroupModel.getParameterModels()).thenReturn(asList(operationParameter, anotherOperationParameter));

    when(operationModel.getParameterGroupModels()).thenReturn(singletonList(parameterGroupModel));
    mockParameter(configurationParameter, SOME_VALUE_PROVIDER_MP_NO_PARAMS);
    mockParameter(operationParameter, SOME_VALUE_PROVIDER_MP_NO_PARAMS);
    mockAnotherParameter(anotherOperationParameter, STRING_TYPE);
  }

  @Test
  void valueProviderShouldBeInstantiable() {
    mockParameter(operationParameter, fromValueProviderClass(NonInstantiableProvider.class), "anotherId");

    validate();
    assertProblems("The Value Provider [NonInstantiableProvider] is not instantiable but it should");
  }

  @Test
  void parameterShouldExist() throws NoSuchFieldException {
    when(operationParameter.getModelProperty(ValueProviderFactoryModelProperty.class))
        .thenReturn(of(withInjectableField(SomeValueProvider.class, "someParam")));

    validate();
    assertProblems("The Value Provider [SomeValueProvider] declares to use a parameter 'someParam' which doesn't exist in the operation 'superOperation'");
  }

  @Test
  void parameterShouldBeOfSameType() throws NoSuchFieldException {
    mockParameter(operationParameter, withInjectableField(SomeOtherValueProvider.class, "someName"), "otherValueProviderId");

    validate();
    assertProblems("The Value Provider [SomeOtherValueProvider] defines a parameter 'someName' of type 'class java.lang.Integer' but in the operation 'superOperation' is of type 'class java.lang.String'");
  }

  @Test
  void injectConnectionInConnectionLessComponent() throws NoSuchFieldException {
    FieldElement fieldElement = new FieldWrapper(SomeValueProvider.class.getDeclaredField("connection"), loader);
    ValueProviderFactoryModelProperty mp =
        new ValueProviderFactoryModelProperty(new JavaValueProviderFactory(SomeValueProvider.class, emptyList(), null,
                                                                           fieldElement),
                                              emptyList());
    when(operationParameter.getModelProperty(ValueProviderFactoryModelProperty.class)).thenReturn(of(mp));
    when(operationParameter.getValueProviderModel())
        .thenReturn(of(new ValueProviderModel(emptyList(), false, true, true, 1, "name", "valueProviderId")));

    validate();
    assertProblems("The Value Provider [SomeValueProvider] defines that requires a connection, but is used in the operation 'superOperation' which is connection less");
  }

  @Test
  void configurationBasedValueProviderDoesntSupportConnectionInjection() throws NoSuchFieldException {
    FieldElement fieldElement = new FieldWrapper(SomeValueProvider.class.getDeclaredField("connection"), loader);
    ValueProviderFactoryModelProperty mp =
        new ValueProviderFactoryModelProperty(new JavaValueProviderFactory(SomeValueProvider.class, emptyList(), null,
                                                                           fieldElement),
                                              emptyList());
    mockParameter(configurationParameter, mp);
    when(configurationParameter.getValueProviderModel())
        .thenReturn(of(new ValueProviderModel(emptyList(), false, true, true, 1, "name", "valueProviderId")));

    validate();
    assertProblems("The Value Provider [SomeValueProvider] defines that requires a connection which is not allowed for a Value Provider of a configuration's parameter [SomeConfig]");
  }

  @Test
  void configurationBasedValueProviderDoesntSupportConfigurationInjection() throws NoSuchFieldException {
    FieldElement fieldElement = new FieldWrapper(SomeValueProviderRequiringConfig.class.getDeclaredField("config"), loader);
    ValueProviderFactoryModelProperty mp =
        new ValueProviderFactoryModelProperty(new JavaValueProviderFactory(SomeValueProviderRequiringConfig.class, emptyList(),
                                                                           fieldElement,
                                                                           null),
                                              emptyList());
    mockParameter(configurationParameter, mp);
    when(configurationParameter.getValueProviderModel())
        .thenReturn(of(new ValueProviderModel(emptyList(), true, false, true, 1, "name", "requiringConfigValueProviderId")));

    validate();
    assertProblems("The Value Provider [SomeValueProviderRequiringConfig] defines that requires a configuration which is not allowed for a Value Provider of a configuration's parameter [SomeConfig]");
  }

  @Test
  void parameterWithValueProviderShouldBeOfStringType() {
    when(operationParameter.getType()).thenReturn(NUMBER_TYPE);

    validate();
    assertProblems("The parameter [someName] of the operation 'superOperation' is not of String type. Parameters that provides Values should be of String type.");
  }

  @Test
  void parameterWithValueProviderHasRepeatedIdInCompileTime() {
    mockParameter(operationParameter, fromValueProviderClass(SomeOtherValueProvider.class));

    validate();
    assertProblems("The following ValueProvider implementations [org.mule.runtime.module.extension.internal.loader.validation.JavaValueProviderModelValidatorTestCase$SomeValueProvider, org.mule.runtime.module.extension.internal.loader.validation.JavaValueProviderModelValidatorTestCase$SomeOtherValueProvider] use the same id [valueProviderId]. ValueProvider ids must be unique.");
  }

  @Test
  void boundParameterExists() throws NoSuchFieldException {
    when(operationParameter.getModelProperty(ValueProviderFactoryModelProperty.class))
        .thenReturn(of(withInjectableField(SomeValueProvider.class, "someParam", "someName")));

    validate();
    assertNoErrors();
  }

  @Test
  void boundParameterShouldExist() throws NoSuchFieldException {
    when(operationParameter.getModelProperty(ValueProviderFactoryModelProperty.class))
        .thenReturn(of(withInjectableField(SomeValueProvider.class, "someParam", "anotherName")));

    validate();
    assertProblems("The Value Provider [SomeValueProvider] declares to use a parameter 'anotherName' which doesn't exist in the operation 'superOperation'");
  }

  @Test
  void boundParameterFromExtractionExpressionExists() throws NoSuchFieldException {
    when(operationParameter.getModelProperty(ValueProviderFactoryModelProperty.class))
        .thenReturn(of(withInjectableField(SomeValueProvider.class, "someParam", "someName.someTag.@attribute")));

    validate();
    assertNoErrors();
  }

  @Test
  void boundParameterFromExtractionExpressionShouldExist() throws NoSuchFieldException {
    when(operationParameter.getModelProperty(ValueProviderFactoryModelProperty.class))
        .thenReturn(of(withInjectableField(SomeValueProvider.class, "someParam", "anotherName.nested.fields")));

    validate();
    assertProblems("The Value Provider [SomeValueProvider] declares to use a parameter 'anotherName' which doesn't exist in the operation 'superOperation'");
  }

  @Test
  void parameterWithValueProviderHasDifferentIdInCompileTime() {
    mockParameter(operationParameter, fromValueProviderClass(SomeOtherValueProvider.class), "anotherId");

    validate();
    assertNoErrors();
  }

  @Test
  void parameterShouldNotBeAnnotatedWithBothOfValuesAndFieldValues() {
    mockParameter(operationParameter, fromValueProviderClass(SomeOtherValueProvider.class));

    Map<String, ValueProviderFactoryModelProperty> fieldsValueProviderFactories =
        singletonMap("simple.path", fromValueProviderClass(SomeOtherValueProvider.class));
    FieldsValueProviderFactoryModelProperty fieldsValueProviderFactoryModelProperty =
        new FieldsValueProviderFactoryModelProperty(fieldsValueProviderFactories);

    when(operationParameter.getModelProperty(FieldsValueProviderFactoryModelProperty.class))
        .thenReturn(of(fieldsValueProviderFactoryModelProperty));

    validate();
    assertProblems("Parameter [someName] from operation with name superOperation has both a Value Provider and a Field Value Provider");
  }

  @Test
  void parameterWithFieldValueProviderDoNotHaveToBeStringType() {
    when(operationParameter.getModelProperty(ValueProviderFactoryModelProperty.class)).thenReturn(empty());

    Map<String, ValueProviderFactoryModelProperty> valueProviderFactoryModelPropertyBuilders =
        singletonMap("simple.path", fromValueProviderClass(SomeOtherValueProvider.class));

    mockParameter(operationParameter, valueProviderFactoryModelPropertyBuilders);

    validate();
    assertNoErrors();
  }

  @Test
  void modelTypeMatchesValueProviderTypeForStringParameter() throws NoSuchFieldException {
    mockAnotherParameter(anotherOperationParameter, STRING_TYPE_WITH_ANNOTATIONS);
    when(operationParameter.getModelProperty(ValueProviderFactoryModelProperty.class))
        .thenReturn(of(withInjectableField(SomeValueProvider.class, "anotherParameter")));

    validate();
    assertNoErrors();
  }

  @Test
  void modelTypeMatchesValueProviderTypeForNumberParameter() throws NoSuchFieldException {
    mockAnotherParameter(anotherOperationParameter, NUMBER_TYPE_WITH_ANNOTATIONS);
    when(operationParameter.getModelProperty(ValueProviderFactoryModelProperty.class))
        .thenReturn(of(withInjectableField(SomeValueProvider.class, "anotherParameter")));
    validate();
    assertNoErrors();
  }

  @Test
  void modelTypeMatchesValueProviderTypeForObjectParameter() throws NoSuchFieldException {
    mockAnotherParameter(anotherOperationParameter, OBJECT_TYPE_WITH_ANNOTATIONS);
    when(operationParameter.getModelProperty(ValueProviderFactoryModelProperty.class))
        .thenReturn(of(withInjectableField(SomeValueProvider.class, "anotherParameter")));
    validate();
    assertNoErrors();
  }

  @Test
  void modelTypeMatchesValueProviderTypeForArrayParameter() throws NoSuchFieldException {
    mockAnotherParameter(anotherOperationParameter, ARRAY_TYPE_WITH_ANNOTATIONS);
    when(operationParameter.getModelProperty(ValueProviderFactoryModelProperty.class))
        .thenReturn(of(withInjectableField(SomeValueProvider.class, "anotherParameter")));
    validate();
    assertNoErrors();
  }

  @Test
  void modelTypeContainsTwoParametersWithTheSameName() {
    ParameterModel clashingParameterMock = mock(ParameterModel.class);

    mockParameter(clashingParameterMock, SOME_VALUE_PROVIDER_MP_NO_PARAMS, "anotherValueProviderId", "anotherParameter",
                  OBJECT_TYPE);
    when(clashingParameterMock.getModelProperty(ValueProviderFactoryModelProperty.class)).thenReturn(empty());
    when(operationModel.getAllParameterModels())
        .thenReturn(asList(operationParameter, anotherOperationParameter, clashingParameterMock));
    when(parameterGroupModel.getParameterModels())
        .thenReturn(asList(operationParameter, anotherOperationParameter, clashingParameterMock));

    validate();
    assertProblems("Parameter [someName] from operation with name superOperation has a Value Provider defined, but that operation has one or more parameters with repeated names [anotherParameter]. Components with parameters with non-unique names do not support Value Providers");
  }

  private void assertProblems(String errorMessage) {
    List<String> actualErrorMessages = problemsReporter.getErrors().stream().map(Problem::getMessage).toList();
    assertThat(actualErrorMessages, contains(errorMessage));
  }

  private void assertNoErrors() {
    List<Problem> errors = problemsReporter.getErrors();
    assertThat(errors, IsEmptyCollection.empty());
  }

  private void validate() {
    valueProviderModelValidator.validate(extensionModel, problemsReporter);
  }

  private void mockParameter(ParameterModel parameter, ValueProviderFactoryModelProperty valueProviderFactoryModelProperty) {
    mockParameter(parameter, valueProviderFactoryModelProperty, "valueProviderId");
  }

  private void mockParameter(ParameterModel parameter, ValueProviderFactoryModelProperty valueProviderFactoryModelProperty,
                             String valueProviderId) {
    mockParameter(parameter, valueProviderFactoryModelProperty, valueProviderId, "someName", STRING_TYPE);
  }

  private void mockParameter(ParameterModel parameter, ValueProviderFactoryModelProperty valueProviderFactoryModelProperty,
                             String valueProviderId,
                             String paramName, MetadataType type) {
    when(parameter.getModelProperty(ValueProviderFactoryModelProperty.class)).thenReturn(of(valueProviderFactoryModelProperty));
    when(parameter.getModelProperty(ImplementingParameterModelProperty.class)).thenReturn(empty());
    when(parameter.getModelProperty(DeclaringMemberModelProperty.class)).thenReturn(empty());
    when(parameter.getName()).thenReturn(paramName);
    when(parameter.getType()).thenReturn(type);
    when(parameter.getValueProviderModel())
        .thenReturn(of(new ValueProviderModel(valueProviderFactoryModelProperty.getActingParameterModels(), false, false, true, 1,
                                              "name", valueProviderId)));
  }

  private void mockParameter(ParameterModel parameter,
                             Map<String, ValueProviderFactoryModelProperty> valueProviderFactoryModelProperties) {

    FieldsValueProviderFactoryModelProperty fieldsValueProviderFactoryModelProperty =
        new FieldsValueProviderFactoryModelProperty(valueProviderFactoryModelProperties);

    FieldValueProviderModel fieldValueProviderModel =
        new FieldValueProviderModel(emptyList(), false, false, true, 1, "name", "providerId", "simple.path");
    List<FieldValueProviderModel> fieldValueProviderModels = singletonList(fieldValueProviderModel);

    when(operationParameter.getModelProperty(FieldsValueProviderFactoryModelProperty.class))
        .thenReturn(of(fieldsValueProviderFactoryModelProperty));
    when(parameter.getModelProperty(ImplementingParameterModelProperty.class)).thenReturn(empty());
    when(parameter.getModelProperty(DeclaringMemberModelProperty.class)).thenReturn(empty());
    when(parameter.getName()).thenReturn("someName");
    when(parameter.getType()).thenReturn(OBJECT_TYPE);
    when(parameter.getFieldValueProviderModels()).thenReturn(fieldValueProviderModels);
  }

  private void mockAnotherParameter(ParameterModel parameter, MetadataType type) {
    when(parameter.getModelProperty(ImplementingParameterModelProperty.class)).thenReturn(empty());
    when(parameter.getModelProperty(DeclaringMemberModelProperty.class)).thenReturn(empty());
    when(parameter.getName()).thenReturn("anotherParameter");
    when(parameter.getType()).thenReturn(type);
  }

  private ActingParameterModel toActingParameterModel(FieldElement fieldElement) {
    return toActingParameterModel(fieldElement, fieldElement.getName());
  }

  private ActingParameterModel toActingParameterModel(FieldElement fieldElement, String extractionExpression) {
    return new ImmutableActingParameterModel(fieldElement.getName(), fieldElement.isRequired(), extractionExpression);
  }

  private static ValueProviderFactoryModelProperty fromValueProviderClass(Class<?> providerClass) {
    ValueProviderFactory valueProviderFactory = new JavaValueProviderFactory(providerClass, emptyList(), null, null);
    return new ValueProviderFactoryModelProperty(valueProviderFactory, emptyList());
  }

  private ValueProviderFactoryModelProperty withInjectableField(Class<?> providerClass, String fieldName)
      throws NoSuchFieldException {
    FieldElement fieldElement = new FieldWrapper(providerClass.getDeclaredField(fieldName), loader);
    ActingParameterModel actingParameterModel = toActingParameterModel(fieldElement);
    ValueProviderFactory valueProviderFactory =
        new JavaValueProviderFactory(providerClass, singletonList(fieldElement), null, null);
    return new ValueProviderFactoryModelProperty(valueProviderFactory, singletonList(actingParameterModel));
  }

  private ValueProviderFactoryModelProperty withInjectableField(Class<?> providerClass, String fieldName,
                                                                String extractionExpression)
      throws NoSuchFieldException {
    FieldElement fieldElement = new FieldWrapper(providerClass.getDeclaredField(fieldName), loader);
    ActingParameterModel actingParameterModel = toActingParameterModel(fieldElement, extractionExpression);
    ValueProviderFactory valueProviderFactory =
        new JavaValueProviderFactory(providerClass, singletonList(fieldElement), null, null);
    return new ValueProviderFactoryModelProperty(valueProviderFactory, singletonList(actingParameterModel));
  }

  public static class SomeValueProvider implements ValueProvider {

    @Parameter
    String someParam;

    @Parameter
    InputStream anotherParameter;

    @Connection
    String connection;

    @Override
    public Set<Value> resolve() {
      return emptySet();
    }
  }

  public static class SomeOtherValueProvider implements ValueProvider {

    @Parameter
    Integer someName;

    @Connection
    String connection;

    @Override
    public Set<Value> resolve() {
      return emptySet();
    }
  }

  public static class SomeValueProviderRequiringConfig implements ValueProvider {

    @Config
    String config;

    @Override
    public Set<Value> resolve() {
      return emptySet();
    }
  }

  public static class NonInstantiableProvider implements ValueProvider {

    private NonInstantiableProvider() {}

    @Override
    public Set<Value> resolve() throws ValueResolvingException {
      return null;
    }
  }

  private Map<Class<? extends TypeAnnotation>, TypeAnnotation> createStringAnnotations() {
    Map<Class<? extends TypeAnnotation>, TypeAnnotation> annotations = new HashMap<>();
    annotations.put(ClassInformationAnnotation.class, new ClassInformationAnnotation(InputStream.class));
    annotations.put(EnumAnnotation.class, new EnumAnnotation<>(new String[] {"value1", "value2"}));
    return annotations;
  }

  private Map<Class<? extends TypeAnnotation>, TypeAnnotation> createNumberAnnotations() {
    Map<Class<? extends TypeAnnotation>, TypeAnnotation> annotations = new HashMap<>();
    annotations.put(ClassInformationAnnotation.class, new ClassInformationAnnotation(InputStream.class));
    annotations.put(EnumAnnotation.class, new EnumAnnotation<>(new Number[] {1, 2}));
    return annotations;
  }

  private Map<Class<? extends TypeAnnotation>, TypeAnnotation> createObjectAnnotations() {
    Map<Class<? extends TypeAnnotation>, TypeAnnotation> annotations = new HashMap<>();
    annotations.put(ClassInformationAnnotation.class, new ClassInformationAnnotation(InputStream.class));
    return annotations;
  }
}
