/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.extension.internal.loader.parser.java;

import static org.mule.runtime.module.extension.internal.loader.parser.java.MuleExtensionAnnotationParser.mapReduceSingleAnnotation;
import static org.mule.runtime.module.extension.internal.loader.parser.java.source.JavaSourceModelParserUtils.fromSdkBackPressureMode;

import static java.util.stream.Collectors.toCollection;

import org.mule.runtime.extension.api.annotation.source.BackPressure;
import org.mule.runtime.extension.api.loader.parser.BackPressureModeParser;
import org.mule.runtime.extension.api.runtime.source.BackPressureMode;
import org.mule.runtime.module.extension.api.loader.java.type.AnnotationValueFetcher;
import org.mule.runtime.module.extension.api.loader.java.type.SourceElement;
import org.mule.runtime.module.extension.internal.loader.parser.java.source.JavaSourceModelParserUtils;

import java.util.LinkedHashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;

/**
 * Implementation of {@link BackPressureModeParser} for Java based syntax
 *
 * @since 4.10.0
 */
public class JavaBackPressureModeParser implements BackPressureModeParser {

  /**
   * Creates a {@link JavaBackPressureModeParser} by introspecting the {@link BackPressure} and
   * {@link org.mule.sdk.api.annotation.source.BackPressure} annotations in the Source class.
   *
   * @param sourceElement the element that defines the Source.
   * @return the {@link JavaBackPressureModeParser}.
   */
  public static Optional<JavaBackPressureModeParser> fromBackPressureAnnotation(SourceElement sourceElement) {
    return mapReduceSingleAnnotation(sourceElement, "source", sourceElement.getName(),
                                     BackPressure.class,
                                     org.mule.sdk.api.annotation.source.BackPressure.class,
                                     JavaBackPressureModeParser::fromExtApiFetcher,
                                     JavaBackPressureModeParser::fromSdkApiFetcher);
  }

  private static JavaBackPressureModeParser fromExtApiFetcher(AnnotationValueFetcher<BackPressure> annotation) {
    return new JavaBackPressureModeParser(annotation.getEnumValue(BackPressure::defaultMode),
                                          annotation.getEnumArrayValue(BackPressure::supportedModes));
  }

  private static JavaBackPressureModeParser fromSdkApiFetcher(AnnotationValueFetcher<org.mule.sdk.api.annotation.source.BackPressure> annotation) {
    return new JavaBackPressureModeParser(annotation.getEnumValue(org.mule.sdk.api.annotation.source.BackPressure::defaultMode),
                                          annotation
                                              .getEnumArrayValue(org.mule.sdk.api.annotation.source.BackPressure::supportedModes));
  }

  private final BackPressureMode defaultMode;
  private final Set<BackPressureMode> supportedModes;

  private JavaBackPressureModeParser(BackPressureMode defaultMode,
                                     List<BackPressureMode> supportedModes) {
    this.defaultMode = defaultMode;
    this.supportedModes = new LinkedHashSet<>(supportedModes);
  }

  private JavaBackPressureModeParser(org.mule.sdk.api.runtime.source.BackPressureMode defaultMode,
                                     List<org.mule.sdk.api.runtime.source.BackPressureMode> supportedModes) {
    this.defaultMode = fromSdkBackPressureMode(defaultMode);
    this.supportedModes = supportedModes.stream()
        .map(JavaSourceModelParserUtils::fromSdkBackPressureMode)
        .collect(toCollection(LinkedHashSet::new));
  }

  @Override
  public BackPressureMode getDefaultMode() {
    return defaultMode;
  }

  @Override
  public Set<BackPressureMode> getSupportedModes() {
    return supportedModes;
  }
}
