/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.container.internal;

import static org.mule.runtime.container.internal.ExportedServiceMatcher.like;

import static java.nio.file.Files.copy;
import static java.nio.file.Files.createDirectory;
import static java.util.Collections.singletonList;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.contains;
import static org.hamcrest.Matchers.containsInAnyOrder;

import org.mule.runtime.module.artifact.api.classloader.ExportedService;
import org.mule.runtime.module.artifact.internal.util.FileJarExplorer;
import org.mule.tck.junit4.AbstractMuleTestCase;
import org.mule.tck.size.SmallTest;
import org.mule.tck.util.CompilerUtils;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.junit.jupiter.api.BeforeAll;
import org.junit.jupiter.api.Test;
import org.junit.jupiter.api.io.TempDir;

@SmallTest
class JreExplorerTestCase extends AbstractMuleTestCase {

  private static final String FOO_SERVICE_PATH = "META-INF/services/org.foo.FooService";
  private static final String SERVICE_RESOURCE = "META-INF/fooResource.txt";
  private static final String FOO_RESOURCE = "fooResource.txt";
  private static final String FOO_JAR_FILENAME = "foo.jar";
  private static final String FOO_SERVICE_INTERFACE = "org.foo.FooService";
  private static final String BAR_SERVICE_INTERFACE = "org.bar.BarService";
  private static final String FOO_PACKAGE = "org.foo";
  private static final String BAR_PACKAGE = "org.bar";
  private static final String LIB_FOLDER = "lib";
  private static final String SUB_FOLDER = "subFolder";
  private static final String BAR_SERVICE_PATH = "META-INF/services/org.bar.BarService";
  private static final String BAR_RESOURCE = "barResource.txt";
  private static final String BAR_JAR_FILENAME = "bar.jar";
  private static final String RESOURCE_PATH = "/resource.txt";


  private static Path fooJar;

  private static Path barJar;

  @TempDir
  public Path temporaryFolder;

  @BeforeAll
  public static void beforeClass() throws URISyntaxException, IOException {
    fooJar = new CompilerUtils.JarCompiler()
        .compiling(Path.of(JreExplorerTestCase.class.getResource("/org/foo/Foo.java").toURI()))
        .including(Path.of(JreExplorerTestCase.class.getResource(RESOURCE_PATH).toURI()), FOO_RESOURCE)
        .including(Path.of(JreExplorerTestCase.class.getResource(RESOURCE_PATH).toURI()), SERVICE_RESOURCE)
        .including(Path.of(JreExplorerTestCase.class.getResource(RESOURCE_PATH).toURI()), FOO_SERVICE_PATH)
        .compile(FOO_JAR_FILENAME);

    barJar = new CompilerUtils.JarCompiler()
        .compiling(Path.of(JreExplorerTestCase.class.getResource("/org/bar/Bar.java").toURI()))
        .including(Path.of(JreExplorerTestCase.class.getResource(RESOURCE_PATH).toURI()), BAR_RESOURCE)
        .including(Path.of(JreExplorerTestCase.class.getResource(RESOURCE_PATH).toURI()), BAR_SERVICE_PATH)
        .compile(BAR_JAR_FILENAME);
  }

  @Test
  void readsJar() throws Exception {

    Path libFolder = temporaryFolder.resolve(LIB_FOLDER);
    createDirectory(libFolder);
    Path innerFooJar = libFolder.resolve(FOO_JAR_FILENAME);
    copy(fooJar, innerFooJar);

    List<String> paths = singletonList(libFolder.toAbsolutePath().toString());

    Set<String> packages = new HashSet<>();
    Set<String> resources = new HashSet<>();
    List<ExportedService> services = new ArrayList<>();
    JreExplorer.explorePaths(paths, packages, resources, services);

    assertThat(packages, contains(FOO_PACKAGE));
    assertThat(resources, containsInAnyOrder(FOO_RESOURCE, SERVICE_RESOURCE));
    assertThat(services, contains(like(FOO_SERVICE_INTERFACE, getServiceResourceUrl(innerFooJar, FOO_SERVICE_PATH))));
  }

  @Test
  void readsJarInFolder() throws Exception {

    Path libFolder = temporaryFolder.resolve(LIB_FOLDER);
    createDirectory(libFolder);
    Path subFolder = libFolder.resolve(SUB_FOLDER);
    createDirectory(subFolder);
    Path innerFooJar = subFolder.resolve(FOO_JAR_FILENAME);
    copy(fooJar, innerFooJar);

    List<String> paths = singletonList(libFolder.toAbsolutePath().toString());

    Set<String> packages = new HashSet<>();
    Set<String> resources = new HashSet<>();
    List<ExportedService> services = new ArrayList<>();
    JreExplorer.explorePaths(paths, packages, resources, services);

    assertThat(packages, contains(FOO_PACKAGE));
    assertThat(resources, containsInAnyOrder(FOO_RESOURCE, SERVICE_RESOURCE));
    assertThat(services, contains(like(FOO_SERVICE_INTERFACE, getServiceResourceUrl(innerFooJar, FOO_SERVICE_PATH))));
  }

  @Test
  void multipleJars() throws Exception {
    Path libFolder = temporaryFolder.resolve(LIB_FOLDER);
    createDirectory(libFolder);
    Path subFolder = libFolder.resolve(SUB_FOLDER);
    createDirectory(subFolder);
    Path innerFooJar = subFolder.resolve(FOO_JAR_FILENAME);
    copy(fooJar, innerFooJar);
    Path innerBarJar = subFolder.resolve(BAR_JAR_FILENAME);
    copy(barJar, innerBarJar);

    List<String> paths = singletonList(libFolder.toAbsolutePath().toString());

    Set<String> packages = new HashSet<>();
    Set<String> resources = new HashSet<>();
    List<ExportedService> services = new ArrayList<>();
    JreExplorer.explorePaths(paths, packages, resources, services);

    assertThat(packages, containsInAnyOrder(FOO_PACKAGE, BAR_PACKAGE));
    assertThat(resources, containsInAnyOrder(FOO_RESOURCE, SERVICE_RESOURCE, BAR_RESOURCE));
    assertThat(services, containsInAnyOrder(like(FOO_SERVICE_INTERFACE, getServiceResourceUrl(innerFooJar, FOO_SERVICE_PATH)),
                                            like(BAR_SERVICE_INTERFACE, getServiceResourceUrl(innerBarJar, BAR_SERVICE_PATH))));
  }

  private URL getServiceResourceUrl(Path resourceFile, String serviceInterface) throws MalformedURLException {
    return FileJarExplorer.getServiceResourceUrl(resourceFile.toUri().toURL(), serviceInterface);
  }
}
