/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.runtime.module.artifact.activation.internal.descriptor;

import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;

import org.mule.tck.junit4.AbstractMuleTestCase;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.nio.charset.StandardCharsets;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TemporaryFolder;

import io.qameta.allure.Issue;

/**
 * Test case for {@link XmlMuleConfigurationsFilter} to verify handling of XML files with and without UTF-8 BOM.
 */
public class XmlMuleConfigurationsFilterTestCase extends AbstractMuleTestCase {

  private static final String VALID_MULE_CONFIG_XML = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
      + "<mule xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"\n"
      + "      xmlns=\"http://www.mulesoft.org/schema/mule/core\" xmlns:spring=\"http://www.springframework.org/schema/beans\"\n"
      + "      xsi:schemaLocation=\"http://www.mulesoft.org/schema/mule/core http://www.mulesoft.org/schema/mule/core/current/mule.xsd\">\n"
      + "</mule>";

  private static final String VALID_MULE_DOMAIN_CONFIG_XML = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
      + "<mule-domain xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\"\n"
      + "      xmlns=\"http://www.mulesoft.org/schema/mule/domain\"\n"
      + "      xsi:schemaLocation=\"http://www.mulesoft.org/schema/mule/domain http://www.mulesoft.org/schema/mule/domain/current/mule-domain.xsd\">\n"
      + "</mule-domain>";

  private static final String INVALID_XML = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
      + "<invalid-root xmlns=\"http://www.example.org/schema/invalid\">\n"
      + "</invalid-root>";

  // UTF-8 BOM bytes: EF BB BF
  private static final byte[] UTF8_BOM = {(byte) 0xEF, (byte) 0xBB, (byte) 0xBF};

  @Rule
  public TemporaryFolder temporaryFolder = new TemporaryFolder();

  private XmlMuleConfigurationsFilter filter;

  @Before
  public void setUp() {
    filter = new XmlMuleConfigurationsFilter();
  }

  @Test
  public void testValidMuleConfigWithoutBOM() throws IOException {
    File configFile = createFileWithoutBOM("valid-mule-config.xml", VALID_MULE_CONFIG_XML);
    assertThat("Expected valid mule config to be accepted", filter.filter(configFile), is(true));
  }

  @Test
  public void testValidMuleDomainConfigWithoutBOM() throws IOException {
    File configFile = createFileWithoutBOM("valid-domain-config.xml", VALID_MULE_DOMAIN_CONFIG_XML);
    assertThat("Expected valid mule domain config to be accepted", filter.filter(configFile), is(true));
  }

  @Test
  public void testInvalidConfigWithoutBOM() throws IOException {
    File configFile = createFileWithoutBOM("invalid-config.xml", INVALID_XML);
    assertThat("Expected invalid config to be rejected", filter.filter(configFile), is(false));
  }

  @Test
  public void testNonXmlFile() throws IOException {
    File nonXmlFile = new File(temporaryFolder.getRoot(), "not-xml.txt");
    nonXmlFile.createNewFile();
    assertThat("Expected non-XML file to be rejected", filter.filter(nonXmlFile), is(false));
  }

  /**
   * This test demonstrates the failure when parsing XML files with UTF-8 BOM. The current implementation uses FileReader which
   * doesn't handle BOM correctly, causing the XML parser to fail.
   */
  @Test
  @Issue("W-20084951")
  public void testValidMuleConfigWithBOM() throws IOException {
    File configFile = createFileWithBOM("mule-config-with-bom.xml", VALID_MULE_CONFIG_XML);
    assertThat("Expected valid mule config with BOM to be accepted", filter.filter(configFile), is(true));
  }

  /**
   * This test demonstrates the failure when parsing mule domain XML files with UTF-8 BOM.
   */
  @Test
  @Issue("W-20084951")
  public void testValidMuleDomainConfigWithBOM() throws IOException {
    File configFile = createFileWithBOM("domain-config-with-bom.xml", VALID_MULE_DOMAIN_CONFIG_XML);
    assertThat("Expected valid mule domain config with BOM to be accepted", filter.filter(configFile), is(true));
  }

  /**
   * Test that even with BOM, invalid configs should still be rejected.
   */
  @Test
  public void testInvalidConfigWithBOM() throws IOException {
    File configFile = createFileWithBOM("invalid-config-with-bom.xml", INVALID_XML);
    // This should be false regardless of BOM, but currently might fail for different reasons
    assertThat("Expected invalid config with BOM to be rejected", filter.filter(configFile), is(false));
  }

  @Test
  public void testInvalidXmlFile() throws IOException {
    File configFile = new File(temporaryFolder.getRoot(), "invalid-1.txt");
    // This should be false regardless of BOM, but currently might fail for different reasons
    assertThat("Expected invalid config with BOM to be rejected", filter.filter(configFile), is(false));
  }

  /**
   * Creates a file with UTF-8 BOM at the beginning.
   */
  private File createFileWithBOM(String filename, String content) throws IOException {
    File file = new File(temporaryFolder.getRoot(), filename);
    try (FileOutputStream fos = new FileOutputStream(file)) {
      // Write BOM first
      fos.write(UTF8_BOM);
      // Write content as UTF-8 bytes
      fos.write(content.getBytes(StandardCharsets.UTF_8));
    }
    return file;
  }

  /**
   * Creates a file without BOM (normal UTF-8).
   */
  private File createFileWithoutBOM(String filename, String content) throws IOException {
    File file = new File(temporaryFolder.getRoot(), filename);
    try (OutputStreamWriter writer = new OutputStreamWriter(new FileOutputStream(file), StandardCharsets.UTF_8)) {
      writer.write(content);
    }
    return file;
  }
}

