/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.raml.internal.fragments.handler;

import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.api.model.MetadataFormat;
import org.raml.v2.api.model.v10.datamodel.TypeDeclaration;

import java.util.Arrays;
import java.util.List;

public class TypeDeclarationHandlerManager {

  private final MetadataFormat metadataFormat;
  private List<? extends TypeDeclarationHandler> typeDeclarationHandlers =
      Arrays.asList(
                    new NilTypeDeclarationHandler(),
                    new BooleanTypeDeclarationHandler(),
                    new StringTypeDeclarationHandler(),
                    new IntegerTypeDeclarationHandler(),
                    new NumberTypeDeclarationHandler(),
                    new DateTimeTypeDeclarationHandler(),
                    new DateTimeOnlyTypeDeclarationHandler(),
                    new DateOnlyTypeDeclarationHandler(),
                    new TimeOnlyTypeDeclarationHandler(),
                    new FileTypeDeclarationHandler(),
                    new ArrayTypeDeclarationHandler(),
                    new ObjectTypeDeclarationHandler(),
                    new UnionTypeDeclarationHandler(),
                    new XMLTypeDeclarationHandler(),
                    new JSONTypeDeclarationHandler(),
                    new ExternalTypeDeclarationHandler(),
                    new AnyTypeDeclarationHandler());

  public TypeDeclarationHandlerManager(MetadataFormat metadataFormat) {
    this.metadataFormat = metadataFormat;
  }

  public TypeBuilder<?> handle(TypeDeclaration typeDeclaration, ParsingContext parsingContext) {
    if (typeDeclaration == null) {
      throw new IllegalArgumentException(String.format("Unhandled null type declaration"));
    }
    if (parsingContext.isTypeBuilderPresent(typeDeclaration)) {
      return parsingContext.getTypeBuilder(typeDeclaration).get();
    } else {
      final BaseTypeBuilder root = BaseTypeBuilder.create(metadataFormat);

      for (TypeDeclarationHandler handler : typeDeclarationHandlers) {
        if (handler.handles(typeDeclaration)) {
          return handler.handle(typeDeclaration, root, this, parsingContext);
        }
      }

      throw new IllegalArgumentException(String.format("Unhandled type declaration %s", typeDeclaration.type()));
    }
  }
}
