/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.raml.api.map;

import static org.mule.metadata.api.model.MetadataFormat.JAVA;
import org.mule.metadata.api.TypeLoader;
import org.mule.metadata.api.model.ArrayType;
import org.mule.metadata.api.model.BooleanType;
import org.mule.metadata.api.model.DateTimeType;
import org.mule.metadata.api.model.DateType;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.NumberType;
import org.mule.metadata.api.model.ObjectFieldType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.model.StringType;
import org.mule.metadata.raml.internal.CustomHandlingTypeDeclarationTypeLoader;
import org.mule.metadata.raml.internal.RamlTypeLoader;

import java.io.File;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;

/**
 * Loader of a MAP (aka: dynamic object) from a RAML file with a few restrictions. The obtained {@link MetadataType} from the
 * RAML must be of only some valid field types, see {@link #VALID_FIELD_TYPES}
 *
 * @since 1.0
 */
public class MapTypeLoader implements TypeLoader {

  private static final List<Class<? extends MetadataType>> VALID_FIELD_TYPES =
      Arrays.asList(StringType.class, NumberType.class, BooleanType.class,
                    DateType.class, DateTimeType.class);

  private final RamlTypeLoader ramlTypeLoader;

  /**
   * Constructs a CSV type loader from a RAML document
   *
   * @param ramlFile file pointing to the RAML resource
   */
  public MapTypeLoader(File ramlFile) {
    this.ramlTypeLoader = new RamlTypeLoader(ramlFile, new CustomHandlingTypeDeclarationTypeLoader(JAVA));
  }

  /**
   * Constructs a CSV type loader from a RAML document
   *
   * @param content content of the RAML resource
   * @param ramlLocation pointer to the RAML location
   */
  public MapTypeLoader(String content, String ramlLocation) {
    this.ramlTypeLoader = new RamlTypeLoader(content, ramlLocation, new CustomHandlingTypeDeclarationTypeLoader(JAVA));
  }

  @Override
  public Optional<MetadataType> load(String typeIdentifier) {
    return load(typeIdentifier, null);
  }

  @Override
  public Optional<MetadataType> load(String typeIdentifier, String typeAlias) {
    return ramlTypeLoader.load(typeIdentifier, typeAlias).map(metadataType -> {
      validate(metadataType);
      return metadataType;
    });
  }

  private boolean validArrayType(MetadataType metadataType) {
    if (!(metadataType instanceof ArrayType)) {
      return false;
    }

    ArrayType arrayType = (ArrayType) metadataType;
    final MetadataType elementType = arrayType.getType();

    return validSimpleType(elementType) || validObjectType(elementType);
  }

  private boolean validObjectType(MetadataType metadataType) {
    if (!(metadataType instanceof ObjectType)) {
      return false;
    }

    ObjectType objectType = (ObjectType) metadataType;
    for (ObjectFieldType objectFieldType : objectType.getFields()) {
      if (!(validSimpleType(objectFieldType) || validObjectType(objectFieldType) || validArrayType(objectFieldType))) {
        return false;
      }
    }
    return true;
  }

  private boolean validSimpleType(MetadataType metadataType) {
    for (Class<? extends MetadataType> validSimpleTypeClass : VALID_FIELD_TYPES) {
      if (validSimpleTypeClass.isAssignableFrom(metadataType.getClass())) {
        return true;
      }
    }
    return false;
  }


  private void validate(MetadataType metadataType) {
    if (validArrayType(metadataType) || validObjectType(metadataType)) {
      throw new RuntimeException(
                                 "Invalid RAML type for Map format.");
    }
  }

}
