/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.persistence.type.adapter;

import static java.util.Arrays.asList;
import static java.util.stream.Collectors.toList;

import org.mule.metadata.api.annotation.TypeAnnotation;

import com.google.gson.Gson;
import com.google.gson.TypeAdapter;
import com.google.gson.stream.JsonReader;
import com.google.gson.stream.JsonWriter;

import java.io.IOException;
import java.lang.reflect.Constructor;
import java.lang.reflect.Field;
import java.lang.reflect.Modifier;
import java.util.Arrays;
import java.util.List;

/**
 * {@link TypeAdapter} implementation that serializes {@link TypeAnnotation} instances that have only one field. If a class
 * with more than one field is used, the serialization will fail. This {@link TypeAdapter} exist to override the default
 * {@link Gson} behaviour to generate a more clean JSON.
 * <p>
 * <b>Default GSON:</b>
 * <pre>{"typeId":{"value" : "org.mule.metadata.SomeClass"}}</pre>
 *
 * <b>Using a OnlyOneFieldTypeAdapter:</b>
 * <pre>{"typeId':"org.mule.metadata.SomeClass"}</pre>
 *
 * @param <T> {@link TypeAnnotation} class
 * @since 1.0
 */
final class OnlyOneFieldTypeAdapter<T> extends TypeAdapter<T> {

  private static final int FIRST_PARAMETER = 0;
  private static final int FIRST_FIELD = 0;
  private final Gson gson;
  private final Class<T> annotationType;
  private Field fieldToSerialize;

  OnlyOneFieldTypeAdapter(Gson gson, Class<T> annotationType) {
    this.gson = gson;
    this.annotationType = annotationType;

    fieldToSerialize = getFieldToSerialize(annotationType);
  }

  @Override
  public void write(JsonWriter out, T value) throws IOException {
    try {
      gson.toJson(getFieldValue(value, fieldToSerialize), fieldToSerialize.getType(), out);
    } catch (IllegalAccessException e) {
      throw new RuntimeException(String.format("An unexpected error occurred serializing %s", value), e);
    }
  }

  @Override
  public T read(JsonReader in) throws IOException {
    try {
      final Constructor<T> declaredConstructor = annotationType.getDeclaredConstructor(fieldToSerialize.getType());
      final Object field = gson.fromJson(in, fieldToSerialize.getType());
      return declaredConstructor.newInstance(field);
    } catch (Exception e) {
      throw new RuntimeException(String.format("An unexpected error occurred serializing %s", fieldToSerialize), e);
    }
  }

  private Object getFieldValue(Object annotation, Field annotationField) throws IllegalAccessException {
    annotationField.setAccessible(true);
    return annotationField.get(annotation);
  }

  private Field getFieldToSerialize(Class<T> annotationType) {
    final Field fieldToSerialize;

    List<Field> instanceFields = asList(annotationType.getDeclaredFields())
        .stream()
        .filter((field) -> !Modifier.isStatic(field.getModifiers()))
        .collect(toList());


    if (instanceFields.size() == 1) {
      fieldToSerialize = instanceFields.get(FIRST_FIELD);
    } else {
      throw new IllegalArgumentException(String.format("The class '%s' contains more than one serializable field",
                                                       annotationType.getName()));
    }

    final long constructorsCount = Arrays.asList(annotationType.getDeclaredConstructors())
        .stream()
        .filter(constructor -> constructor.getParameterCount() == 1)
        .filter(constructor -> constructor.getParameters()[FIRST_PARAMETER].getType().equals(fieldToSerialize.getType()))
        .count();

    if (constructorsCount != 1) {
      throw new IllegalArgumentException(String.format(
                                                       "The class '%s' requires a constructor with an unique parameter of type '%s",
                                                       annotationType.getName(), fieldToSerialize));
    }

    return fieldToSerialize;
  }
}
