/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.json.api.utils;

import static java.util.stream.Collectors.toList;

import static org.slf4j.LoggerFactory.getLogger;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;

import org.everit.json.schema.Schema;
import org.everit.json.schema.SchemaException;
import org.everit.json.schema.loader.SchemaLoader;
import org.everit.json.schema.loader.internal.DefaultSchemaClient;
import org.json.JSONArray;
import org.json.JSONObject;
import org.json.JSONTokener;
import org.slf4j.Logger;

public class SchemaHelper {

  private static final Logger LOGGER = getLogger(SchemaHelper.class);

  public static List<URI> getDependencies(File schemaFile) throws IOException {
    Set<URI> dependencies = new LinkedHashSet<>();
    try (InputStream inputStream = new FileInputStream(schemaFile)) {
      final JSONObject rawSchema = new JSONObject(new JSONTokener(inputStream));
      generateSchema(rawSchema, schemaFile.toURI(), schemaLoaderBuilder -> {
        schemaLoaderBuilder.schemaClient(new DefaultSchemaClient() {

          @Override
          public InputStream get(String url) {
            dependencies.add(URI.create(url));
            return super.get(url);
          }
        });
      });

      if (localReferences(rawSchema)) {
        dependencies.add(schemaFile.toURI());
      }
    }
    return dependencies.stream().collect(toList());
  }

  private static boolean localReferences(final JSONObject rawSchema) {
    if (rawSchema.keySet().contains("$ref")) {
      final String ref = rawSchema.getString("$ref");
      if (ref.startsWith("#/")) {
        return true;
      }
    }

    for (String key : rawSchema.keySet()) {
      final Object object = rawSchema.get(key);
      if (object instanceof JSONArray) {
        for (Object item : (JSONArray) object) {
          if (item instanceof JSONObject && localReferences((JSONObject) item)) {
            return true;
          }
        }
      } else if (object instanceof JSONObject) {
        if (localReferences((JSONObject) object)) {
          return true;
        }
      }
    }

    return false;
  }

  public static Schema generateSchema(JSONObject rawSchema, URI baseUri,
                                      final Consumer<SchemaLoader.SchemaLoaderBuilder> schemaLoaderBuilderConsumer) {
    final SchemaLoader.SchemaLoaderBuilder schemaLoaderBuilder = new SchemaLoader.SchemaLoaderBuilder();
    Optional.ofNullable(baseUri).ifPresent(uri -> {
      schemaLoaderBuilder.resolutionScope(baseUri);
      if (schemaLoaderBuilderConsumer != null) {
        schemaLoaderBuilderConsumer.accept(schemaLoaderBuilder);
      }
    });
    schemaLoaderBuilder.schemaJson(rawSchema);

    SchemaLoader schemaLoader;
    try {
      schemaLoader = schemaLoaderBuilder.build();
    } catch (SchemaException e) {
      if ("#: could not determine version".equals(e.getMessage())) {
        LOGGER.warn("json-schema doesn't declare a version. Defaulting to draft-v7...");
        schemaLoaderBuilder.draftV7Support();
        schemaLoader = schemaLoaderBuilder.build();
      } else {
        throw e;
      }
    }

    return schemaLoader.load().build();
  }


  public static Schema generateSchema(JSONObject rawSchema, URI baseUri) {
    return generateSchema(rawSchema, baseUri, null);
  }

}
