/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.json.api.utils;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import org.everit.json.schema.Schema;
import org.everit.json.schema.loader.SchemaLoader;
import org.everit.json.schema.loader.internal.DefaultSchemaClient;
import org.json.JSONObject;
import org.json.JSONTokener;

public class SchemaHelper {

  public static List<URI> getDependencies(File schemaFile) throws IOException {
    Set<URI> dependencies = new LinkedHashSet<>();
    try (InputStream inputStream = new FileInputStream(schemaFile)) {
      final JSONObject rawSchema = new JSONObject(new JSONTokener(inputStream));
      generateSchema(rawSchema, schemaFile.toURI(), schemaLoaderBuilder -> {
        schemaLoaderBuilder.httpClient(new DefaultSchemaClient() {

          @Override
          public InputStream get(String url) {
            dependencies.add(URI.create(url));
            return super.get(url);
          }
        });
      });
    }
    return dependencies.stream().collect(Collectors.toList());
  }

  public static Schema generateSchema(JSONObject rawSchema, URI baseUri,
                                      final Consumer<SchemaLoader.SchemaLoaderBuilder> schemaLoaderBuilderConsumer) {
    final SchemaLoader.SchemaLoaderBuilder schemaLoaderBuilder = new SchemaLoader.SchemaLoaderBuilder();
    Optional.ofNullable(baseUri).ifPresent(uri -> {
      schemaLoaderBuilder.resolutionScope(baseUri);
      if (schemaLoaderBuilderConsumer != null) {
        schemaLoaderBuilderConsumer.accept(schemaLoaderBuilder);
      }
    });
    schemaLoaderBuilder.schemaJson(rawSchema);
    final SchemaLoader schemaLoader = schemaLoaderBuilder.build();
    return schemaLoader.load().build();
  }


  public static Schema generateSchema(JSONObject rawSchema, URI baseUri) {
    return generateSchema(rawSchema, baseUri, null);
  }

}
