/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.json.api.handler;

import static org.mule.metadata.internal.utils.StringUtils.isNotEmpty;
import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.ObjectFieldTypeBuilder;
import org.mule.metadata.api.builder.ObjectTypeBuilder;
import org.mule.metadata.api.builder.TypeBuilder;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Pattern;
import java.util.stream.Collectors;

import org.everit.json.schema.ObjectSchema;
import org.everit.json.schema.Schema;

public class ObjectHandler implements SchemaHandler {

  @Override
  public boolean handles(Schema schema) {
    return schema instanceof ObjectSchema;
  }

  @Override
  public TypeBuilder<?> handle(Schema schema, BaseTypeBuilder root, HandlerManager handlerManager,
                               ParsingContext parsingContext) {

    final ObjectSchema objectSchema = (ObjectSchema) schema;
    final ObjectTypeBuilder objectMetadataBuilder = root.objectType();
    parsingContext.registerBuilder(objectSchema, objectMetadataBuilder);

    final String description = objectSchema.getDescription();
    if (isNotEmpty(description)) {
      objectMetadataBuilder.description(description);
    }

    final String title = objectSchema.getTitle();
    if (isNotEmpty(title)) {
      objectMetadataBuilder.label(title);
    }

    final String id = objectSchema.getId();
    if (isNotEmpty(id)) {
      objectMetadataBuilder.id(id);
    }

    final Set<Map.Entry<String, Schema>> properties = objectSchema.getPropertySchemas().entrySet();
    final List<String> requiredProperties = objectSchema.getRequiredProperties();
    // Add named properties
    for (Map.Entry<String, Schema> property : properties) {
      final ObjectFieldTypeBuilder field = objectMetadataBuilder.addField().key(property.getKey());
      field.required(requiredProperties.contains(property.getKey()));
      final Schema value = property.getValue();
      field.value(handlerManager.handle(value, parsingContext));
    }
    final Map<Pattern, Schema> patternProperties = objectSchema.getPatternProperties();
    // Sort them by pattern
    final Collection<Map.Entry<Pattern, Schema>> entries =
        patternProperties.entrySet().stream().sorted((o1, o2) -> o1.getKey().toString().compareTo(o2.getKey().toString()))
            .collect(Collectors.toList());
    for (Map.Entry<Pattern, Schema> patternSchemaEntry : entries) {
      final ObjectFieldTypeBuilder field = objectMetadataBuilder.addField();
      field.key(patternSchemaEntry.getKey());
      final Schema value = patternSchemaEntry.getValue();
      field.value(handlerManager.handle(value, parsingContext));
    }
    if (objectSchema.permitsAdditionalProperties()) {
      objectMetadataBuilder.openWith(handlerManager.handle(objectSchema.getSchemaOfAdditionalProperties(), parsingContext));
    }
    return objectMetadataBuilder;
  }
}
