/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.json.api.handler;

import static java.util.stream.Collectors.toList;

import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.IntersectionTypeBuilder;
import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.api.builder.UnionTypeBuilder;

import java.util.Collection;

import org.everit.json.schema.CombinedSchema;
import org.everit.json.schema.EnumSchema;
import org.everit.json.schema.Schema;
import org.everit.json.schema.StringSchema;

public class CombinedHandler implements SchemaHandler {

  @Override
  public boolean handles(Schema schema) {
    return schema instanceof CombinedSchema;
  }

  @Override
  public TypeBuilder<?> handle(Schema schema, BaseTypeBuilder root, HandlerManager handlerManager,
                               ParsingContext parsingContext) {
    final CombinedSchema combinedSchema = (CombinedSchema) schema;
    Collection<Schema> subschemas = combinedSchema.getSubschemas();

    // filter out redundant StringSchema for enums.
    if (subschemas.stream().anyMatch(subSchema -> subSchema instanceof EnumSchema)) {
      subschemas = subschemas.stream()
          .filter(subSchema -> !(subSchema instanceof StringSchema))
          .collect(toList());
    }

    if (subschemas.size() == 1) {
      return handlerManager.handle(subschemas.iterator().next(), parsingContext);
    }


    if (combinedSchema.getCriterion() == CombinedSchema.ANY_CRITERION
        || combinedSchema.getCriterion() == CombinedSchema.ONE_CRITERION) {
      final UnionTypeBuilder unionType = root.unionType();
      for (Schema subschema : subschemas) {
        unionType.of(handlerManager.handle(subschema, parsingContext));
      }
      return unionType;
    } else if (combinedSchema.getCriterion() == CombinedSchema.ALL_CRITERION) {
      final IntersectionTypeBuilder intersectionType = root.intersectionType();
      for (Schema subschema : subschemas) {
        intersectionType.of(handlerManager.handle(subschema, parsingContext));
      }
      return intersectionType;
    }
    return root.anyType();
  }
}
