/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.java.internal.handler;

import org.mule.metadata.api.builder.BaseTypeBuilder;
import org.mule.metadata.api.builder.ObjectTypeBuilder;
import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.java.api.annotation.ClassInformationAnnotation;
import org.mule.metadata.java.api.handler.ClassHandler;
import org.mule.metadata.java.api.handler.TypeHandlerManager;
import org.mule.metadata.java.api.utils.ParsingContext;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.lang.reflect.TypeVariable;
import java.util.List;
import java.util.Map;
import java.util.Optional;

public class MapClassHandler implements ClassHandler {

  private static final Type[] EMPTY_ARRAY = new Type[0];

  @Override
  public boolean handles(Class<?> clazz) {
    return Map.class.isAssignableFrom(clazz);
  }

  @Override
  public TypeBuilder<?> handleClass(Class<?> clazz,
                                    List<Type> genericTypes, TypeHandlerManager typeHandlerManager,
                                    ParsingContext context, BaseTypeBuilder typeBuilder) {

    final ObjectTypeBuilder objectTypeBuilder = typeBuilder.objectType();

    Type type = getMapValueGeneric(clazz, genericTypes.toArray(new Type[genericTypes.size()]));

    if (mapValueShouldBeAnyType(type)) {
      objectTypeBuilder.open();
    } else {
      openWith(typeHandlerManager, context, objectTypeBuilder, type);
    }
    objectTypeBuilder.with(new ClassInformationAnnotation(clazz, genericTypes));
    return objectTypeBuilder;
  }

  private boolean mapValueShouldBeAnyType(Type type) {
    return type == null || (type instanceof TypeVariable
        && ((((TypeVariable) type).getBounds().length == 0) || ((TypeVariable) type).getBounds()[0].equals(Object.class)));
  }

  private Type getMapValueGeneric(Class<?> clazz, Type[] genericTypes) {
    // If the class is Map, return the second generic Type
    if (clazz.equals(Map.class)) {
      return genericTypes.length == 0 ? null : genericTypes[1];
    }

    TypeVariable[] unresolvedGenerics = clazz.getTypeParameters();
    // Check if superclass is still instance of Map, if so, call recursively to superclass
    if (clazz.getSuperclass() != null && Map.class.isAssignableFrom(clazz.getSuperclass())) {
      return getMapValueGenericFromSuperClass(clazz, genericTypes, unresolvedGenerics);
    }

    // Return the generic used in Map for the implemented interface that is or extends from Map
    return getMapValueGenericFromImplementedInterface(clazz, genericTypes, unresolvedGenerics);
  }

  private Type getMapValueGenericFromImplementedInterface(Class<?> clazz, Type[] genericTypes,
                                                          TypeVariable[] unresolvedGenerics) {
    int interfaceIndex;
    Class[] interfaces = clazz.getInterfaces();

    for (interfaceIndex = 0; interfaceIndex < interfaces.length; interfaceIndex++) {
      if (Map.class.isAssignableFrom(interfaces[interfaceIndex])) {
        break;
      }
    }

    Type implementedInterface = clazz.getGenericInterfaces()[interfaceIndex];
    return getMapValueGenericFromGenericType(interfaces[interfaceIndex], implementedInterface, genericTypes, unresolvedGenerics);
  }

  private Type getMapValueGenericFromSuperClass(Class<?> clazz, Type[] genericTypes, TypeVariable[] unresolvedGenerics) {
    Class superClazz = clazz.getSuperclass();
    Type genericSuperClazz = clazz.getGenericSuperclass();
    return getMapValueGenericFromGenericType(superClazz, genericSuperClazz, genericTypes, unresolvedGenerics);
  }

  private Type getMapValueGenericFromGenericType(Class<?> clazz, Type genericType, Type[] genericTypes,
                                                 TypeVariable[] unresolvedGenerics) {
    Type returnType = getMapValueGeneric(clazz, (genericType instanceof ParameterizedType)
        ? ((ParameterizedType) (genericType)).getActualTypeArguments()
        : EMPTY_ARRAY);
    return resolveGenericType(genericTypes, unresolvedGenerics, returnType);
  }

  private Type resolveGenericType(Type[] resolvedGenericTypes, TypeVariable[] unresolvedGenerics, Type typeToResolve) {
    // If already resolved, return it
    if (!(typeToResolve instanceof TypeVariable)) {
      return typeToResolve;
    }
    int resolutionIndex = 0;
    for (Type unResolvedGeneric : unresolvedGenerics) {
      if (unResolvedGeneric.equals(typeToResolve)) {
        break;
      }
      resolutionIndex++;
    }
    // If generics are not resolved, return Object.class
    if (resolutionIndex >= unresolvedGenerics.length) {
      return Object.class;
    }
    // resolvedGenericTypes will be empty if generics are not resolved, in this case, the unresolvedType must be returned
    return resolvedGenericTypes.length == 0 ? unresolvedGenerics[resolutionIndex] : resolvedGenericTypes[resolutionIndex];
  }

  private void openWith(TypeHandlerManager typeHandlerManager, ParsingContext context, ObjectTypeBuilder objectTypeBuilder,
                        Type type) {
    final Optional<TypeBuilder<?>> keyTypeBuilder = context.getTypeBuilder(type);
    if (keyTypeBuilder.isPresent()) {
      objectTypeBuilder.openWith(keyTypeBuilder.get());
    } else {
      typeHandlerManager.handle(type, context, objectTypeBuilder.openWith());
    }
  }

}
