/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.annotation;

import static java.util.Collections.singletonList;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.containsInAnyOrder;
import static org.hamcrest.Matchers.hasSize;
import static org.hamcrest.core.Is.is;
import org.mule.metadata.annotation.pojo.InstantiableClass;
import org.mule.metadata.annotation.pojo.MultipleConstructor;
import org.mule.metadata.annotation.pojo.NoDefaultConstructor;
import org.mule.metadata.java.api.annotation.ClassInformationAnnotation;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Map;
import java.util.Set;

import org.junit.Test;

public class ClassInformationAnnotationTestCase {

  private ClassInformationAnnotation annotation;

  @Test
  public void isFinalClass() {
    annotation = new ClassInformationAnnotation(InstantiableClass.class, null);
    assertThat(annotation.isFinal(), is(true));
  }

  @Test
  public void isAbstractClass() {
    annotation = new ClassInformationAnnotation(AbstractClass.class, null);
    assertThat(annotation.isAbstract(), is(true));
    assertThat(annotation.isInstantiable(), is(false));
  }

  @Test
  public void isInterface() {
    annotation = new ClassInformationAnnotation(Interface.class, null);
    assertThat(annotation.isInterface(), is(true));
    assertThat(annotation.isInstantiable(), is(false));
  }

  @Test
  public void isMap() {
    annotation = new ClassInformationAnnotation(CustomMap.class, null);
    assertThat(annotation.isMap(), is(true));

    annotation = new ClassInformationAnnotation(InstantiableClass.class, null);
    assertThat(annotation.isMap(), is(false));
  }

  @Test
  public void hasDefaultConstructor() {
    annotation = new ClassInformationAnnotation(InstantiableClass.class, null);
    assertThat(annotation.isInstantiable(), is(true));
    assertThat(annotation.hasDefaultConstructor(), is(true));

    annotation = new ClassInformationAnnotation(MultipleConstructor.class, null);
    assertThat(annotation.isInstantiable(), is(true));
    assertThat(annotation.hasDefaultConstructor(), is(true));

    annotation = new ClassInformationAnnotation(NoDefaultConstructor.class, null);
    assertThat(annotation.isInstantiable(), is(false));
    assertThat(annotation.hasDefaultConstructor(), is(false));
  }

  @Test
  public void getParent() {
    annotation = new ClassInformationAnnotation(WithParent.class, null);
    assertThat(annotation.getParent().isPresent(), is(true));
    assertThat(annotation.getParent().get(), is(AbstractClass.class.getName()));
  }

  @Test
  public void getImplementedInterfaces() {
    annotation = new ClassInformationAnnotation(Interface.class, null);
    assertThat(annotation.getImplementedInterfaces(),
               containsInAnyOrder(Serializable.class.getName(), Cloneable.class.getName()));

    annotation = new ClassInformationAnnotation(NoDefaultConstructor.class, null);
    assertThat(annotation.getImplementedInterfaces(),
               containsInAnyOrder(Serializable.class.getName(), Cloneable.class.getName()));
  }

  @Test
  public void getGenerics() {
    annotation = new ClassInformationAnnotation(Object.class, singletonList(String.class));
    assertThat(annotation.getGenericTypes(), hasSize(1));
    assertThat(annotation.getGenericTypes().get(0), is(String.class.getName()));

    annotation = new ClassInformationAnnotation(Object.class, Arrays.asList(String.class, Object.class));
    assertThat(annotation.getGenericTypes(), hasSize(2));
    assertThat(annotation.getGenericTypes(), containsInAnyOrder(String.class.getName(), Object.class.getName()));
  }

  private static abstract class AbstractClass {
  }

  private static class WithParent extends AbstractClass {
  }

  private interface Interface extends Cloneable, Serializable {
  }

  private static class CustomMap implements Map {

    @Override
    public int size() {
      return 0;
    }

    @Override
    public boolean isEmpty() {
      return false;
    }

    @Override
    public boolean containsKey(Object key) {
      return false;
    }

    @Override
    public boolean containsValue(Object value) {
      return false;
    }

    @Override
    public Object get(Object key) {
      return null;
    }

    @Override
    public Object put(Object key, Object value) {
      return null;
    }

    @Override
    public Object remove(Object key) {
      return null;
    }

    @Override
    public void putAll(Map m) {

    }

    @Override
    public void clear() {

    }

    @Override
    public Set keySet() {
      return null;
    }

    @Override
    public Collection values() {
      return null;
    }

    @Override
    public Set<Entry> entrySet() {
      return null;
    }
  }

}
