/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.ast.internal;

import static java.util.Arrays.stream;
import static java.util.Collections.emptyList;
import static java.util.Collections.unmodifiableList;
import static java.util.stream.Collectors.toList;
import static javax.lang.model.element.Modifier.ABSTRACT;
import static javax.lang.model.element.Modifier.FINAL;
import static javax.lang.model.element.Modifier.PUBLIC;
import static javax.lang.model.type.TypeKind.NONE;

import org.mule.metadata.java.api.annotation.ClassInformationAnnotation;

import javax.annotation.processing.ProcessingEnvironment;
import javax.lang.model.element.ElementKind;
import javax.lang.model.element.ExecutableElement;
import javax.lang.model.element.TypeElement;
import javax.lang.model.type.ArrayType;
import javax.lang.model.type.DeclaredType;
import javax.lang.model.type.PrimitiveType;
import javax.lang.model.type.TypeMirror;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Factory to create {@link ClassInformationAnnotation} from the Java AST
 *
 * @since 1.1.0
 */
public class ClassInformationAnnotationFactory {

  /**
   * This is done because I didn't find a way to obtain the TypeMirror of a "Plain Array", the implemented interfaces of an
   * String[] TypeMirror are not the same of a String[] Class.
   */
  private static final List<String> arrayInterfaces =
      stream(String[].class.getInterfaces()).map(Class::getCanonicalName).collect(toList());
  private static Map<String, Class> primitiveTypeClasses = new HashMap<String, Class>() {

    {
      put("int", Integer.TYPE);
      put("long", Long.TYPE);
      put("double", Double.TYPE);
      put("float", Float.TYPE);
      put("boolean", Boolean.TYPE);
      put("char", Character.TYPE);
      put("byte", Byte.TYPE);
      put("void", Void.TYPE);
      put("short", Short.TYPE);
    }
  };

  public static ClassInformationAnnotation fromTypeMirror(TypeMirror typeMirror, ProcessingEnvironment processingEnvironment) {
    if (typeMirror instanceof ArrayType) {
      return fromArrayType((ArrayType) typeMirror);
    } else if (typeMirror instanceof PrimitiveType) {
      return fromPrimitiveType((PrimitiveType) typeMirror);
    } else if (typeMirror instanceof DeclaredType) {
      return fromDeclaredType((DeclaredType) typeMirror, processingEnvironment);
    } else {
      throw new IllegalArgumentException("The given TypeMirror is not supported. Type: " + typeMirror.getClass());
    }
  }

  private static ClassInformationAnnotation fromPrimitiveType(PrimitiveType typeMirror) {
    return new ClassInformationAnnotation(primitiveTypeClasses.get(typeMirror.toString()));
  }

  private static ClassInformationAnnotation fromArrayType(ArrayType arrayType) {
    String classname = arrayType.toString().replace(".NestedClassType", "$NestedClassType");
    return new ClassInformationAnnotation(classname, false, false, false, true, true,
                                          unmodifiableList(arrayInterfaces), "", emptyList(),
                                          false);
  }

  private static ClassInformationAnnotation fromDeclaredType(DeclaredType declaredType,
                                                             ProcessingEnvironment processingEnvironment) {
    TypeElement typeElement = (TypeElement) declaredType.asElement();

    String name = typeElement.getQualifiedName().toString();
    String classname;
    try {
      classname = Class.forName(name).getName();
    } catch (ClassNotFoundException e) {
      classname = name.replace(".NestedClassType", "$NestedClassType");
    }

    List<String> implementedInterfaces = getImplementedInterfaces(processingEnvironment, typeElement);
    String parent = getParentClass(processingEnvironment, typeElement);
    boolean isFinal = typeElement.getModifiers().contains(FINAL);
    boolean isAbstract = typeElement.getModifiers().contains(ABSTRACT);
    boolean isInterface = typeElement.getKind().isInterface();
    boolean hasDefaultConstructor = hasDefaultConstructor(typeElement, isInterface);
    List<String> genericTypes = getGenericTypes(declaredType, processingEnvironment);
    boolean isInstantiable = !isInterface && !isAbstract && hasDefaultConstructor;
    boolean isMap = isMap(processingEnvironment, typeElement);

    return new ClassInformationAnnotation(classname, hasDefaultConstructor, isInterface, isInstantiable, isAbstract, isFinal,
                                          implementedInterfaces, parent, genericTypes, isMap);
  }

  private static List<String> getImplementedInterfaces(ProcessingEnvironment processingEnvironment, TypeElement typeElement) {
    return typeElement.getInterfaces().stream().map(inter -> processingEnvironment.getTypeUtils().asElement(inter))
        .map(element -> processingEnvironment.getElementUtils().getBinaryName((TypeElement) element))
        .map(CharSequence::toString).collect(toList());
  }

  private static String getParentClass(ProcessingEnvironment processingEnvironment, TypeElement typeElement) {
    TypeMirror superclass = typeElement.getSuperclass();

    if (!superclass.getKind().equals(NONE) && !processingEnvironment.getTypeUtils().isSameType(
                                                                                               processingEnvironment
                                                                                                   .getElementUtils()
                                                                                                   .getTypeElement(Object.class
                                                                                                       .getName())
                                                                                                   .asType(),
                                                                                               superclass)) {
      return processingEnvironment.getElementUtils()
          .getBinaryName((TypeElement) processingEnvironment.getTypeUtils().asElement(superclass)).toString();
    } else {
      return "";
    }
  }

  private static List<String> getGenericTypes(DeclaredType declaredType, ProcessingEnvironment processingEnvironment) {
    return declaredType.getTypeArguments().stream()
        .map(typeMirror -> processingEnvironment.getTypeUtils().asElement(typeMirror))
        .filter(elem -> elem instanceof TypeElement)
        .map(elem -> processingEnvironment.getElementUtils().getBinaryName((TypeElement) elem)).map(CharSequence::toString)
        .collect(toList());
  }

  private static boolean isMap(ProcessingEnvironment processingEnvironment, TypeElement typeElement) {
    return processingEnvironment.getTypeUtils()
        .isAssignable(processingEnvironment.getTypeUtils().erasure(typeElement.asType()),
                      processingEnvironment.getElementUtils().getTypeElement(Map.class.getName()).asType());
  }


  private static boolean hasDefaultConstructor(TypeElement typeElement, boolean isInterface) {
    List<ExecutableElement> constructors =
        typeElement.getEnclosedElements().stream().filter(elem -> elem.getKind().equals(ElementKind.CONSTRUCTOR))
            .map(elem -> (ExecutableElement) elem).collect(toList());

    boolean containsDefaultConstructor =
        constructors.stream().filter(elem -> elem.getModifiers().contains(PUBLIC))
            .anyMatch(executableElement -> executableElement.getParameters().size() == 0);

    return !isInterface
        && (constructors.isEmpty() || (containsDefaultConstructor && typeElement.getModifiers().contains(PUBLIC)));
  }
}
