/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.ast.api;

import static javax.lang.model.type.TypeKind.BOOLEAN;

import javax.annotation.processing.ProcessingEnvironment;
import javax.lang.model.element.Element;
import javax.lang.model.element.ElementKind;
import javax.lang.model.element.ExecutableElement;
import javax.lang.model.type.DeclaredType;
import javax.lang.model.type.PrimitiveType;
import javax.lang.model.type.TypeMirror;

import java.util.Collection;
import java.util.HashMap;

/**
 * Utility for AST Types
 *
 * @since 1.1
 */
public final class TypeUtils {

  private TypeUtils() {

  }

  private static final String SET_PREFIX = "set";
  private static final String GET_PREFIX = "get";
  private static final String IS_PREFIX = "is";

  public static Collection<Property> getProperties(Element element, ProcessingEnvironment processingEnvironment) {
    HashMap<String, Property> properties = new HashMap<>();

    element.getEnclosedElements()
        .stream()
        .filter(elem -> elem.getKind().equals(ElementKind.METHOD))
        .map(elem -> (ExecutableElement) elem)
        .forEach(elem -> {
          String methodName = elem.getSimpleName().toString();
          if (methodName.startsWith(GET_PREFIX)) {
            String propertyName = methodName.substring(3);
            addGetter(properties, elem, propertyName);
          } else if (methodName.startsWith(IS_PREFIX) && isBooleanType(elem.getReturnType(), processingEnvironment)) {
            String propertyName = methodName.substring(2);
            addGetter(properties, elem, propertyName);
          } else if (methodName.startsWith(SET_PREFIX)) {
            String propertyName = methodName.substring(3);
            addSetter(properties, elem, propertyName);
          }
        });

    return properties.values();
  }

  private static void addSetter(HashMap<String, Property> properties, ExecutableElement elem, String propertyName) {
    Property property = properties.get(propertyName);
    if (property == null) {
      property = new Property();
      property.setSetterMethod(elem);
      property.setName(propertyName);
      properties.put(propertyName, property);
    } else {
      property.setSetterMethod(elem);
    }
  }

  private static void addGetter(HashMap<String, Property> properties, ExecutableElement elem, String propertyName) {
    Property property = properties.get(propertyName);
    if (property == null) {
      property = new Property();
      property.setGetterMethod(elem);
      property.setName(propertyName);
      properties.put(propertyName, property);
    } else {
      property.setGetterMethod(elem);
    }
  }

  private static boolean isBooleanType(TypeMirror typeMirror, ProcessingEnvironment processingEnvironment) {
    if (typeMirror instanceof PrimitiveType && typeMirror.getKind().equals(BOOLEAN)) {
      return true;
    }

    if (typeMirror instanceof DeclaredType) {
      return processingEnvironment.getTypeUtils()
          .isSameType(typeMirror, processingEnvironment.getElementUtils().getTypeElement(Boolean.class.getName()).asType());
    }

    return false;
  }
}
