/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.ast.internal.handler;

import static javax.lang.model.element.ElementKind.ENUM;
import static javax.lang.model.element.ElementKind.ENUM_CONSTANT;
import static org.mule.metadata.ast.internal.ASTHelper.typeId;

import org.mule.metadata.api.annotation.EnumAnnotation;
import org.mule.metadata.api.builder.TypeBuilder;
import org.mule.metadata.api.model.StringType;
import org.mule.metadata.ast.api.TypeHandler;
import org.mule.metadata.ast.api.IntrospectionContext;

import javax.annotation.processing.ProcessingEnvironment;
import javax.lang.model.element.ElementKind;
import javax.lang.model.type.DeclaredType;
import javax.lang.model.type.TypeMirror;
import javax.lang.model.type.TypeVisitor;

/**
 * {@link TypeHandler} for enum types. (a.k.a.: {@link StringType} with an {@link EnumAnnotation})
 *
 * @since 1.1.0
 */
public class EnumTypeHandler extends BaseTypeHandler {

  public EnumTypeHandler(ProcessingEnvironment processingEnvironment) {
    super(processingEnvironment);
  }

  @Override
  public boolean handles(TypeMirror typeMirror) {
    return typeMirror instanceof DeclaredType && ((DeclaredType) typeMirror).asElement().getKind().equals(ElementKind.ENUM);
  }

  @Override
  public TypeBuilder<?> handle(TypeMirror typeMirror, TypeVisitor<TypeBuilder<?>, IntrospectionContext> visitor,
                               IntrospectionContext context) {
    if (typeMirror instanceof DeclaredType) {
      DeclaredType declaredType = (DeclaredType) typeMirror;
      if (declaredType.asElement().getKind().equals(ENUM)) {
        return builder().stringType().id(typeId(declaredType))
            .enumOf(declaredType.asElement().getEnclosedElements().stream()
                .filter(elem -> elem.getKind().equals(ENUM_CONSTANT))
                .map(elem -> elem.getSimpleName().toString())
                .toArray(String[]::new));
      }
    }

    throw new IllegalArgumentException("The given TypeMirror is not handleable. " + typeMirror);
  }
}
