/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.catalog.internal.builder;

import static org.mule.metadata.api.utils.MetadataTypeUtils.checkArgument;
import static org.mule.metadata.api.utils.MetadataTypeUtils.isNotNull;
import org.mule.metadata.catalog.internal.model.loaders.TypeLoaderFactory;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

public class TypeLoaderRegistry {

  private static class TypeLoaderRegistryKey {

    private final String typeFormat;
    private final String loaderFormat;

    TypeLoaderRegistryKey(String typeFormat, String loaderFormat) {
      checkArgument(isNotNull(typeFormat), "typeFormat cannot be null.");
      checkArgument(isNotNull(loaderFormat), "loaderFormat cannot be null.");
      this.typeFormat = typeFormat;
      this.loaderFormat = loaderFormat;
    }

    @Override
    public boolean equals(Object o) {
      if (this == o)
        return true;
      if (o == null || getClass() != o.getClass())
        return false;
      TypeLoaderRegistryKey that = (TypeLoaderRegistryKey) o;
      return typeFormat.equals(that.typeFormat) && loaderFormat.equals(that.loaderFormat);
    }

    @Override
    public int hashCode() {
      int result = typeFormat.hashCode();
      result = 31 * result + loaderFormat.hashCode();
      return result;
    }
  }

  private final Map<TypeLoaderRegistryKey, TypeLoaderFactory> typeLoaderFactoryMap;

  public TypeLoaderRegistry(List<TypeLoaderFactory> factories) {
    typeLoaderFactoryMap = new HashMap<>();
    factories.stream().forEach(this::register);
  }

  private void register(TypeLoaderFactory typeLoaderFactory) {
    typeLoaderFactoryMap
        .put(new TypeLoaderRegistryKey(typeLoaderFactory.getTypeFormat(), typeLoaderFactory.getLoaderFormat()),
             typeLoaderFactory);
  }

  public Optional<TypeLoaderFactory> getFactory(String typeFormat, String shapeFormat) {
    return Optional.ofNullable(typeLoaderFactoryMap.get(new TypeLoaderRegistryKey(typeFormat, shapeFormat)));
  }
}
