/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.catalog.api;

import org.mule.metadata.api.TypeLoader;
import org.mule.metadata.api.model.BooleanType;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.StringType;
import org.mule.metadata.catalog.internal.loader.TypeResolverLoader;

import java.net.URL;
import java.util.Optional;

/**
 * This object is responsible of looking for a {@link MetadataType} from a given reference by delegating the resolution to the right
 * {@link TypeLoader}, where if found a {@link MetadataType} will be returned.
 * See {@link #resolveType(String)}
 *
 * @since 1.0
 */
public interface TypeResolver {

  String CATALOG_SEPARATOR = "::";

  String ARRAY_SUFFIX = "[]";

  /**
   * Returns the {@link MetadataType} of the specified type if found. The value of {@code reference} must be a conjunction of
   * information where it could have the following pieces
   * <ol>
   *   <li>The type loader's ID</li>
   *   <li>The type identifier that might, or might not, contain a selector</li>
   * </ol>
   * <p/>
   * Lets see some examples on the premise that the {@link TypeResolver} was instantiated by 3 catalogs:
   * <ol>
   *  <li>one from an XML schema named "xml-types", with a targetNamespace="http://uri", and two elements named "User" and "Account"</li>
   *  <li>one from an JSON schema named "json-types", with a definition named "address"</li>
   * </ol>
   * To use them, the {@code reference} must have one of the following values:
   * <ul>
   *   <li>"xml-types::{http://uri}User": to reference the "User" element of the XML schema</li>
   *   <li>"xml-types::{http://uri}Account": to reference the "Account" element of the XML schema</li>
   *   <li>"xml-types::{http://uri}Account[]": to reference an array of "Account" element of the XML schema</li>
   *   <li>"json-types::#/": to reference the complete JSON schema</li>
   *   <li>"json-types::#/address": to reference the "address" type from within the JSON schema</li>
   *   <li>"json-types": similarly to "json-types::#/", we will allow that reference to the type loader's ID will map to the root
   *   element of the schema (if possible). Useful for samples, and JSON schemas</li>
   *   <li>"json-types[]": to reference an array of the type loader's ID will map to the root.</li>
   * </ul>
   * <p/>
   * Besides custom types loaders, the {@link TypeResolver} will held the responsibility of providing a fixed set of primitive
   * types, see {@link PrimitiveTypesTypeLoader#PRIMITIVE_TYPES}. To reference them, the literal name will be enough:
   * <ul>
   *   <li>"string": to reference a {@link StringType}</li>
   *   <li>"boolean": to reference a {@link BooleanType}</li>
   * </ul>
   * and so on..
   *
   * @param reference The identifier name of the type
   * @return The type if found
   * @throws TypeResolverException if the {@code reference} has a type loader's ID that is absent in the current manager.
   */
  Optional<MetadataType> resolveType(String reference) throws TypeResolverException;

  /**
   * Reads the entire file to properly load a {@link TypeResolver}
   *
   * @param classLoader context class loader where the catalog is being created
   * @return a {@link TypeResolver}
   */
  static TypeResolver create(ClassLoader classLoader) {
    return new TypeResolverLoader().load(classLoader);
  }

  /**
   * Reads the entire file to properly load a {@link TypeResolver}
   *
   * @param url targeting the catalog's URLs location
   * @param classLoader context class loader where the catalog is being created
   * @return a {@link TypeResolver}
   */
  static TypeResolver createFrom(URL url, ClassLoader classLoader) {
    return new TypeResolverLoader().load(url, classLoader);
  }

  /**
   * Reads the entire content to properly load a {@link TypeResolver}
   *
   * @param data content of an XML with the catalog's document
   * @param classLoader context class loader where the catalog is being created
   * @return a {@link TypeResolver}
   */
  static TypeResolver createFrom(String data, ClassLoader classLoader) {
    return new TypeResolverLoader().load(data, classLoader);
  }
}
