/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.api.model;

import static java.util.Objects.hash;

import java.util.Objects;

/**
 * Represents a function parameter with a name and a Type
 */
public class FunctionParameter {

  private final String name;
  private final MetadataType type;
  private final boolean optional;

  public FunctionParameter(String name, MetadataType type) {
    this(name, type, false);
  }

  public FunctionParameter(String name, MetadataType type, boolean optional) {
    this.name = name;
    this.type = type;
    this.optional = optional;
  }

  /**
   * Returns the name of the parameter
   *
   * @return The name
   */
  public String getName() {
    return name;
  }

  /**
   * Returns the type of the parameter
   *
   * @return The type of the parameter
   */
  public MetadataType getType() {
    return type;
  }

  /**
   * Returns optionality of the parameter
   *
   * @return Optionality of the parameter
   */
  public boolean isOptional() {
    return optional;
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (!(o instanceof FunctionParameter)) {
      return false;
    }

    FunctionParameter that = (FunctionParameter) o;

    if (optional != that.optional) {
      return false;
    }
    if (!Objects.equals(name, that.name)) {
      return false;
    }
    return Objects.equals(type, that.type);
  }

  @Override
  public int hashCode() {
    return hash(optional, name, type);
  }
}
