/*
 * Copyright 2023 Salesforce, Inc. All rights reserved.
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.api.model.impl;

import static org.mule.metadata.api.utils.MetadataTypeUtils.getTypeId;

import static java.util.Collections.addAll;

import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ObjectFieldType;
import org.mule.metadata.api.model.ObjectKeyType;
import org.mule.metadata.api.model.ObjectType;
import org.mule.metadata.api.visitor.MetadataTypeVisitor;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;

public class DefaultObjectType extends BaseMetadataType implements ObjectType {

  private final Collection<ObjectFieldType> fields;
  private final boolean ordered;
  private final MetadataType openRestriction;
  private Object[] fieldValues;

  public DefaultObjectType(Collection<ObjectFieldType> fields, boolean ordered, MetadataType openRestriction,
                           MetadataFormat metadataFormat,
                           Map<Class<? extends TypeAnnotation>, TypeAnnotation> extensions) {
    super(metadataFormat, extensions);
    this.fields = fields;
    this.ordered = ordered;
    this.openRestriction = openRestriction;
  }


  @Override
  public boolean isOrdered() {
    return ordered;
  }

  @Override
  public Optional<MetadataType> getOpenRestriction() {
    return Optional.ofNullable(openRestriction);
  }

  @Override
  public Collection<ObjectFieldType> getFields() {
    return fields;
  }

  @Override
  public Object[] getFieldValues() {
    if (fieldValues == null) {
      fieldValues = createFieldValuesArray();
    }
    return fieldValues;
  }

  private Object[] createFieldValuesArray() {
    List<Object> fieldValues = new ArrayList<>();
    addAll(fieldValues, super.getFieldValues());
    addAll(fieldValues, fields, ordered, openRestriction);
    return fieldValues.toArray(new Object[fieldValues.size()]);
  }

  @Override
  public Optional<ObjectFieldType> getFieldByName(String propertyName) {
    return fields.stream()
        .filter(propertyType -> {
          final ObjectKeyType key = propertyType.getKey();
          return key.isName() && key.getName().getLocalPart().equals(propertyName);
        }).findFirst();
  }

  @Override
  public void accept(MetadataTypeVisitor visitor) {
    visitor.visitObject(this);
  }

  @Override
  public String toString() {
    return getTypeId(this).orElseGet(() -> super.toString());
  }
}
