/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.api.model.impl;

import static org.apache.commons.lang3.ArrayUtils.addAll;
import static org.mule.metadata.internal.utils.EfficientReflectionEquals.reflectionEquals;

import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.model.ArrayType;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.ReflectionAwareComparable;
import org.mule.metadata.api.visitor.MetadataTypeVisitor;

import java.lang.reflect.Field;
import java.util.Map;
import java.util.Objects;
import java.util.function.Supplier;

public class DefaultArrayType extends BaseMetadataType implements ArrayType {

  final private Supplier<MetadataType> type;

  // The only purpose of this field is to be compared on equals().
  private MetadataType typeValue;

  public DefaultArrayType(Supplier<MetadataType> type, MetadataFormat metadataFormat,
                          Map<Class<? extends TypeAnnotation>, TypeAnnotation> extensions) {
    super(metadataFormat, extensions);
    this.type = type;
  }

  private static final Field[] REFLECTION_FIELDS;
  static {
    try {
      REFLECTION_FIELDS = addAll(BASE_REFLECTION_FIELDS,
                                 DefaultArrayType.class.getDeclaredField("typeValue"));
    } catch (NoSuchFieldException e) {
      throw new RuntimeException("Could not resolve field", e);
    }
  }

  @Override
  public Field[] getReflectionComparableFields() {
    return REFLECTION_FIELDS;
  }

  @Override
  public MetadataType getType() {
    return type.get();
  }

  @Override
  public void accept(MetadataTypeVisitor visitor) {
    visitor.visitArrayType(this);
  }

  /**
   * Set the value of the {@link #typeValue} field.
   */
  private void setTypeValueForReflectionEqualsIfNull() {
    if (Objects.isNull(typeValue)) {
      typeValue = type.get();
    }
  }

  @Override
  public boolean equals(Object obj) {
    if (this == obj) {
      return true;
    }
    if (obj == null || getClass() != obj.getClass()) {
      return false;
    }

    // Workaround to ensure the typeValue field is defined on both objects, to be able to compare it using reflectionEquals.
    setTypeValueForReflectionEqualsIfNull();
    ((DefaultArrayType) obj).setTypeValueForReflectionEqualsIfNull();

    return super.equals(obj);
  }
}
