/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.api.builder;

import org.mule.metadata.api.annotation.DescriptionAnnotation;
import org.mule.metadata.api.annotation.LabelAnnotation;
import org.mule.metadata.api.annotation.LengthAnnotation;
import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.annotation.TypeIdAnnotation;
import org.mule.metadata.api.annotation.UniquesItemsAnnotation;
import org.mule.metadata.api.model.ArrayType;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;
import org.mule.metadata.api.model.impl.DefaultArrayType;
import org.mule.metadata.internal.utils.LazyValue;

import java.util.Optional;
import java.util.concurrent.atomic.AtomicReference;

public class ArrayTypeBuilder extends AbstractBuilder<ArrayType>
    implements TypeBuilder<ArrayType>, WithAnnotation<ArrayTypeBuilder> {

  private Optional<TypeBuilder<?>> arrayType;
  private Optional<MetadataType> typeValue;
  private LazyValue<ArrayType> result = new LazyValue<>();

  protected ArrayTypeBuilder(MetadataFormat format) {
    super(format);
    this.arrayType = Optional.empty();
    this.typeValue = Optional.empty();
  }

  public BaseTypeBuilder of() {
    final BaseTypeBuilder typeBuilder = new BaseTypeBuilder(format);
    this.arrayType = Optional.of(typeBuilder);
    return typeBuilder;
  }

  public ArrayTypeBuilder id(String typeIdentifier) {
    return with(new TypeIdAnnotation(typeIdentifier));
  }

  public ArrayTypeBuilder of(TypeBuilder<?> builder) {
    this.arrayType = Optional.ofNullable(builder);
    return this;
  }

  public ArrayTypeBuilder of(MetadataType typeValue) {
    this.typeValue = Optional.ofNullable(typeValue);
    return this;
  }

  public ArrayTypeBuilder with(TypeAnnotation extension) {
    this.addExtension(extension);
    return this;
  }

  public ArrayTypeBuilder boundary(Number minLength, Number maxLength) {
    return with(new LengthAnnotation(minLength, maxLength));
  }

  public ArrayTypeBuilder description(String lang, String content) {
    return with(new DescriptionAnnotation(content, lang));
  }

  public ArrayTypeBuilder description(String content) {
    return with(new DescriptionAnnotation(content));
  }

  public ArrayTypeBuilder uniques() {
    return with(new UniquesItemsAnnotation());
  }

  public ArrayTypeBuilder label(String label) {
    return with(new LabelAnnotation(label));
  }

  @Override
  public ArrayType build() {
    return typeValue.isPresent() ? new DefaultArrayType(() -> typeValue.get(), format, annotations)
        : buildArrayType(arrayType.orElseThrow(() -> new RuntimeException("Array type was not specified.")));
  }

  private ArrayType buildArrayType(TypeBuilder<?> arrayBuilder) {
    AtomicReference<MetadataType> metadataTypeHolder = new AtomicReference<>();
    boolean needsInit = !result.isDefined();
    ArrayType type = result.get(() -> new DefaultArrayType(metadataTypeHolder::get, format, annotations));

    if (needsInit) {
      MetadataType metadataType = isRecursiveType(arrayBuilder) ? type : arrayBuilder.build();
      metadataTypeHolder.set(metadataType);
    }

    return type;
  }

  private boolean isRecursiveType(TypeBuilder<?> typeBuilder) {
    return typeBuilder.equals(this);
  }
}
