/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.api.model.impl;

import static java.util.Objects.hash;
import static org.apache.commons.lang3.ArrayUtils.addAll;
import static org.mule.metadata.internal.utils.EfficientReflectionEquals.reflectionEquals;
import static org.mule.metadata.internal.utils.EfficientReflectionHashCode.reflectionHashCode;
import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.model.AttributeKeyType;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.visitor.MetadataTypeVisitor;

import java.lang.reflect.Field;
import java.util.Map;
import java.util.Optional;
import java.util.regex.Pattern;

import javax.xml.namespace.QName;

public class DefaultAttributeKeyType extends BaseMetadataType implements AttributeKeyType {

  private final Optional<QName> name;
  private final Optional<Pattern> pattern;

  public DefaultAttributeKeyType(Optional<QName> name, Optional<Pattern> pattern, MetadataFormat metadataFormat,
                                 Map<Class<? extends TypeAnnotation>, TypeAnnotation> extensions) {
    super(metadataFormat, extensions);
    this.name = name;
    this.pattern = pattern;
  }

  @Override
  public QName getName() {
    return name.get();
  }

  @Override
  public Pattern getPattern() {
    return pattern.get();
  }

  @Override
  public boolean isName() {
    return name.isPresent();
  }

  @Override
  public boolean isPattern() {
    return pattern.isPresent();
  }

  @Override
  public void accept(MetadataTypeVisitor visitor) {
    visitor.visitAttributeKey(this);
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (!(o instanceof DefaultAttributeKeyType)) {
      return false;
    }

    return reflectionEquals(this, (DefaultAttributeKeyType) o)
        && equalPattern(pattern, ((DefaultAttributeKeyType) o).pattern);
  }

  private transient boolean hashCalculated;
  private transient int hash;

  @Override
  public int hashCode() {
    if (!hashCalculated) {
      hash = hash(reflectionHashCode(this), pattern.map(x -> x.toString().hashCode()).orElse(null));
      hashCalculated = true;
    }
    return hash;
  }

  boolean equalPattern(Optional<Pattern> thisPattern, Optional<Pattern> thatPattern) {
    if (thisPattern.isPresent() != thatPattern.isPresent()) {
      return false;
    }

    if (!thisPattern.isPresent() || !thatPattern.isPresent()) {
      return true;
    }

    return thisPattern.get().toString().equals(thatPattern.get().toString());
  }

  private static final Field[] REFLECTION_FIELDS;
  static {
    try {
      REFLECTION_FIELDS = addAll(BASE_REFLECTION_FIELDS,
                                 DefaultAttributeKeyType.class.getDeclaredField("name"));
    } catch (NoSuchFieldException e) {
      throw new RuntimeException("Could not resolve field", e);
    }
  }

  @Override
  public Field[] getReflectionComparableFields() {
    return REFLECTION_FIELDS;
  }
}
