/*
 * Copyright (c) MuleSoft, Inc.  All rights reserved.  http://www.mulesoft.com
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.txt file.
 */
package org.mule.metadata.api.model.impl;

import static java.util.Collections.unmodifiableMap;
import static org.apache.commons.lang3.builder.ToStringStyle.MULTI_LINE_STYLE;

import org.mule.metadata.api.annotation.DescriptionAnnotation;
import org.mule.metadata.api.annotation.TypeAnnotation;
import org.mule.metadata.api.model.MetadataFormat;
import org.mule.metadata.api.model.MetadataType;

import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import org.apache.commons.lang3.builder.EqualsBuilder;
import org.apache.commons.lang3.builder.HashCodeBuilder;
import org.apache.commons.lang3.builder.ToStringBuilder;

public abstract class BaseMetadataType implements MetadataType {

  protected final Map<Class<? extends TypeAnnotation>, TypeAnnotation> annotations;
  private final MetadataFormat metadataFormat;

  public BaseMetadataType(MetadataFormat metadataFormat, Map<Class<? extends TypeAnnotation>, TypeAnnotation> annotations) {
    this.metadataFormat = metadataFormat;
    this.annotations = unmodifiableMap(annotations);
  }

  @Override
  public Set<TypeAnnotation> getAnnotations() {
    return new LinkedHashSet<>(annotations.values());
  }

  @Override
  public MetadataFormat getMetadataFormat() {
    return metadataFormat;
  }

  @Override
  public <T extends TypeAnnotation> Optional<T> getAnnotation(Class<T> extension) {
    return Optional.ofNullable((T) annotations.get(extension));
  }

  @Override
  public Optional<String> getDescription() {
    return getAnnotation(DescriptionAnnotation.class).map(DescriptionAnnotation::getValue);
  }

  @Override
  public boolean equals(Object obj) {
    if (!getClass().isInstance(obj)) {
      return false;
    }

    return EqualsBuilder.reflectionEquals(this, obj);
  }

  private transient boolean hashCalculated;
  private transient int hash;

  @Override
  public int hashCode() {
    if (!hashCalculated) {
      hash = HashCodeBuilder.reflectionHashCode(this);
      hashCalculated = true;
    }
    return hash;
  }

  @Override
  public String toString() {
    return ToStringBuilder.reflectionToString(this, MULTI_LINE_STYLE);
  }
}
